/*========================================================================
     "ssulibc" - common library classes
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2016 by A. Miyoshi, Univ. Tokyo
========================================================================*/

#ifndef _SSULIBC_H_
#define _SSULIBC_H_

#include "ssulibb.h"

/*========================================================================
     Common Library Classes
========================================================================*/

/*------------------------------------------------------------------------
     Well with Multiple Reaction Channels
      - corresponds to one isomer and one "rrkmth" output
------------------------------------------------------------------------*/

struct connex {
  int toWellIdx, chan;
};

class ratesOneWell {
 public:                                       // --- data members ---
  double T, p;                  // current temp. [K], press. [Torr],
  double alpha, collfreq;       //  alpha [cm-1], and coll. freq. [s-1]
  double redExpon;              // reduction exponent
  double keigen, ktot, kstab, fstab;
  vector<double> k, f;
  double fpop;                  // fraction of population (MW problem)
};

class well {
 public:                                       // --- data members ---
  int index;
  int posBmat, caWell;
  int isReactant;               // reactant flag for multiwell eigen-prob
  vector<connex> connect;
  string fname;
  string title;
  int offset, lowest, upbound;  // offset, lowest grain, & upper-boundary
  int nchan, grsize;            // number of channels and grain size [cm-1]
  int ngrain, nreacg, nnonrg;   // numbers of grains, non-reactive grains
                                //  & reactive grains
  int actchan;                  // activation channel for chem-activation
  pressList press;
  tempList temp;
  double sigma, eps, redmass;   // in [angstrom], [K], and [amu]
  double alpha1000, alpTex;     // alpha at 1000 K & T-exponent
  double err3;                  // ERR3 to determine band width
  const static double DEF_thRhoGrad;
  const static double DEF_redExpFix;
  const static double MIN_redExp, MAX_redExp, STP_redExp;
  double thRhoGrad;             // threshold rho gradient
  double redExpFix;             // fixed reduction exponent
  vector<double> rho;
  vector<vector<double> > rate;
  ratesOneWell current;
/*
  struct {
    double T, p;                // current temp. [K], press. [Torr],
    double alpha, collfreq;     //  alpha [cm-1], and coll. freq. [s-1]
    double redExpon;            // reduction exponent
    double keigen, ktot, kstab, fstab;
    vector<double> k, f;
    double fpop;                // fraction of population (MW problem)
  } current;
*/
  vector<int> chInf;
  int nOutCh, nStab;
 public:                                   // --- member functions ---
  well();
  void clear();
  int readFromUnimol();
  void printParETP(ostream &os, int vb);  // print ETP parameters
  void printParCA(ostream &os);           // ptint chem-activation pars
  int setCurrentTp(double Tin, double pin);
};

/*------------------------------------------------------------------------
     List of Wells
------------------------------------------------------------------------*/

#define DEFAULT_kEthInt 1.e-64
#define DEFAULT_kEthOut 1.e-64

class ratesAllWells : public vector<ratesOneWell> {
 public:                                       // --- data members ---
  double koutTot;
};

class listRateAW : public vector<ratesAllWells> {};

class listWells : public vector<well> {
 public:                                       // --- data members ---
  int totSize, absTop, nTotCh, nTotOCh, nTotStbW;
  int posReactWell;             // position of the reactant well
  int topCut;                   // number of top grains to be cut
  double koutTot, keigen;
  double kEthInt, kEthOut;      // thresholds for k(E)int and k(E)out
  map<int, int> tableIndices;
 public:                                   // --- member functions ---
  listWells();
  void clear();
  int isValidIndex(int idx);
  int isUsedIndex(int idx);
  int autoIndex();                  // automatically generate an unused index
  int getPos(int idx);
  int add(well &w);
  int readUnimol();                 // read UNIMOL files, check, & align
  void mergedTlist(tempList &tl);         // get merged T-list
  void mergedPlist(pressList &pl);        // get merged p-list
  void printParETP(ostream &os, int vb);  // print ETP parameters
  void printParCA(ostream &os);           // ptint chem-activation pars
  int reduceSize(int maxs);               // reduce the total size
  void setPosBmat();                      // set posBmat in each well
  int checkCA();                          // check for chemical activation
  void setErr3(double e);                 // set err3
  void setAlpPar(double a1k, double aEx); // set alpha parameters
 protected:
  int checkConsistency();       // check consistency of wells
  int checkConnect();           // check for well-connections
  void alignSize();             // align the offset & upbound
  int checkEqkE();              // check for equilib. conditions of k(E)
  void setChanInfo();           // set chInf in wells
  void checkKout();             // check kout
};

/*------------------------------------------------------------------------
     Unimolecular Problem Solver
------------------------------------------------------------------------*/

class umolProb {
 public:                                       // --- data members ---
  enum { MXSIZ =   7000,        // maximum size of the problem
         MXCOL =   2100,        // 30% of MXSIZ
         MXSZ2 =  14000,        // twice MXSIZ
         WMSIZ = 455000,        // DSYEVR work size = 65 * MXSIZ
         WISIZ =  70000 };      // MXSIZ * 10
//
//  enum { MXSIZ =  10000,        // maximum size of the problem
//         MXCOL =   3000,        // 30% of MXSIZ
//         MXSZ2 =  20000,        // twice MXSIZ
//         WMSIZ = 650000,        // DSYEVR work size = 65 * MXSIZ
//         WISIZ = 100000 };      // MXSIZ * 10
//
//  Total memory required for b,z,ab,v,r,s,d,work,ipiv,isuppz,iwork
//   = MXSIZ * ( MXSIZ * 1.3 + 75.5) * 8 / 1024 / 1024 (mb)
//  MXSIZ    memory
//   7000    490 MB
//  10000    998 MB
//  15000   2240 MB = 2.19 GB
//
  enum { Eigen = 0, LinEq = 1, LinSS = 2, ItrSS = 3,
         TimEv = 4, TimDs = 5 } probtype;
  enum { ESNone = 0, ESdsyev = 1, ESdsyevr = 2 } eigensolver;
  enum { TEconst = 0, TEinit = 1 } timeevcond;
  int HPL, RFlxOut;
  int maxCycLinSs, maxCycItrSs;
  int verbose;
  int psiz;                     // problem size
  int esiz;                     // number of calculated eigenvalues
  int noffdiag;                 // number of off-diagonal elements
  int posLNE;                   // pos. of the largest neg. eigenvalue
  double sizeLSDSRW;            // size factor of RWORK for LSODES
  double T, p;                  // temperature [K], pressure [Torr]
  double b[MXSIZ][MXSIZ];
  double z[MXCOL][MXSIZ];
  double v[MXSIZ];
  double r[1][MXSIZ];           // right hand side for DPOSV/DPBSV
  double s[MXSIZ];              // similarity transformation vector S-1
  double d[MXSIZ];              // sum of the down-trans rate below lowest
  double work[WMSIZ];
  int ipiv[MXSIZ];
  int isuppz[MXSZ2];
  int iwork[WISIZ];
 public:                                   // --- member functions ---
  umolProb();
  void clear();
  int initMW(listWells &lw);          // initialization for multi-well problem
  void calcHPLMW(listWells &lw, double Tin);
                                      // set Boltzmann flux
  void setBoltzRflxMWall(listWells &lw, double Tin);
  void setBoltzRflxMWsel(listWells &lw, double Tin);
  void setBoltzRflx1W(well &w, double Tin, int pos0);
                                      // LinSs problem driver
  int driveLinSsMWposv(listWells &lw, double Tin, double pin);
                                      // set matrix for multiple well problem
  int setMatMW(listWells &lw, double Tin, double pin);
                                      // - with inverted sign
  int setMatMWinvSgn(listWells &lw, double Tin, double pin);
                                      // solve eigenvalue problem (multi-well)
  int solveEigenMWdsyevr(listWells &lw);
                                      // solve MW linear equation by dposv
  int solveLinEqMWposv(listWells &lw);
  int solveLinEqMWposv2nd(listWells &lw);
                                      // solve chemical-activation time evolution
  int solveTimeEvMW(listWells &lw, timeList &tl);
                                      // solve dissociation time evolution
  int solveTimeDsMW(listWells &lw, timeList &tl, double T0, listRateAW &lrat);
                                      // F for lsode / lsodes
  void lsdF(int neq, double t, double *y, double *ydot);
  void lsdsJAC(int neq, double t, double *y, int j, int *ian, int *jan,
   double *pdj);                      // JAC for lsodes
  void outputVectorMW(ostream &os, listWells &lw);
  void outputVecTimEvMW(ostream &os, listWells &lw, timeList &tl);
  void printMat(ostream &os, int start, double fact);
  void outEigVal(ostream &os, int n);   // output largest n eigenvalues in v
 protected:
  int drvSetETP(well &w, int pos0, int &noff);          // setETPorb driver
  int setETProb(well &w, int pos0, int &noff);          // set P(E,E') in b
  void setRateSmat(well &w, int pos0);  // set k(E) in b & S-1 in s[]
  void setRate(well &w, int pos0);
  void setSmat(well &w, int pos0);
  void setCrossRate(listWells &lw);     // set non-diagonal k(E)
  void clearRflx();                     // clear r[][]
  void clearV();                        // clear v[]
  void setRflxCPos(well &w, int pos0);  // set +r(E) in r
  void setVflxCPos(well &w, int pos0);  // set +r(E) in v
  void restoreRflx(well &w, int pos0);  // restore r from v[]
  void storeRflx(well &w, int pos0, double stp);        // store r to v[]
  void storeRflxAll(double stp);        // store r to v[]
  void restoreSSG();                    // restore r from v[]
  void storeSSG();                      // store r to v[]
  void storeVtoZ(int pos);              // store v to z
  ratesAllWells &getVrate(listWells &lw);  // get rates
  int symmetMat();                      // make b-mat symmetric
  void applyStoR();                     // apply S to r
  void applyStoV();                     // apply S to v
  void applyB(double *oprd, double *rslt);  // apply B to oprd, result rslt
  void addVect(double *tgt, double *adv);   // add adv to tgt
  int numNonZeroBelem();                // count non-zero elements in B
  void makeMatBanded(well &w, int pos0);        // make b[][] banded
  void invSignB();                      // invert sign of b[][]
  void findLargeNegEigen();             // find the largest neg. eigenvalue
                                        // normalization of g (1-well)
  double normG1W(well &w, int pos0, double *g);
                                        // normalization of g (multi-well)
  double normGMW(listWells &lw, double *g);
                                        // integrate k(E)g (1-well)
  void integkEg1W(well &w, int pos0, double *g, double gnorm);
                                        // integrate k(E)g (multi-well)
  void integkEgMW(listWells &lw, double *g, double tgnorm);
};

#endif

