/*========================================================================
     "ssulibc" - common library classes
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2014 by A. Miyoshi, Univ. Tokyo
                                  created: Sept. 22, 2009 (from gpoplibc)
                                   edited: Dec.   8, 2009
                                   edited: Jan.  25, 2010
        - added kEthInt and kEthOut in listWells
                                   edited: May   23, 2010
        - corrected bugs in umolProb::outputVector1W & outputVectorMW
        - improved efficiency of umolProb::driveLinSsMW
                                   edited: June   7, 2010
        - added listWells::setAlpPar
                                   edited: Aug.  29, 2010
        - added outputs of HPL and Rflx vectors
                                   edited: May   31, 2011
        - added time-dependent solver catime
                                   edited: June   6, 2011
        - added truncate function to dislit
                                   edited: May   22, 2014
        - added time-dependent solver distim
                                   edited: Dec.  16, 2014
        - slightly modified to avoid "asymmetry" & "redExp" problems
                              last edited: Sept.  2, 2016
========================================================================*/

#include "ssulibc.h"

// #include "lapack_dsyev/wcpp_dsyev.h"
#include "lapack_dsyevr/wcpp_dsyevr.h"
// #include "lapack_dsysv/wcpp_dsysv.h"
#include "lapack_dposv/wcpp_dposv.h"
// #include "dlsode/wcpp_dlsode.h"
#include "dlsodes/wcpp_dlsodes.h"

/*========================================================================
     Common Library Classes
========================================================================*/

/*------------------------------------------------------------------------
     Well with Multiple Reaction Channels
      - corresponds to one isomer and one "rrkmth" output
------------------------------------------------------------------------*/

// const double well::DEF_thRhoGrad = 5.;
const double well::DEF_thRhoGrad = 3.;
const double well::DEF_redExpFix = -1.;
const double well::MIN_redExp = 1.;
const double well::MAX_redExp = 3.05;
// const double well::MAX_redExp = 5.05;
const double well::STP_redExp = 0.1;

well::well() { clear(); }                         // --- (constructor) ---

void well::clear() {                                      // --- clear ---
  index = 1; caWell = false; isReactant = false;
  connect.clear();
  fname.erase(); title.erase();
  offset = 0; lowest = 0; actchan = 0; sigma = 0.;
  press.clear(); temp.clear();
  thRhoGrad = DEF_thRhoGrad;
  redExpFix = DEF_redExpFix;
  rho.clear(); rate.clear();
}

int well::readFromUnimol() {                     // --- readFromUnimol ---
  int ich, i, inc, nalpha, ixv, jxv, np, jav, ntemp, ioptht, ioptpr;
  double err1, err2, e0, pr, palmt, tv, wt1, wt2, rhv, corrat, rtv;
  string basn = getBasePart(fname), extn = getExtPart(fname);
  ifstream inf;
  vector<double> rtvl;

  if (fname == "") {
    cout << "No file name is given for well-" << index << ".\n";
    return ERRE;
  }
  if (extn == "") { fname = basn + ".dat"; }
  inf.open(fname.c_str());
  if (!inf)
    { cout << "failed to open [" << fname << "].\n"; return ERRE; }
  if (getLineTrim(inf, title) == ERRE) { return ERRE; }
  inf >> inc >> nchan >> grsize;
  inf >> err1 >> err2 >> err3;
  inf >> e0;
  inf >> nalpha;
  if (nalpha < 0) { alpTex = abs(nalpha) / 2.; }
  else { alpTex = 0.; }
  inf >> alpha1000;
  inf >> ixv >> jxv;
  inf >> np;
  for (i = 0; i < np; i++) { inf >> pr; press.push_back(pr); }
  press.sortElimDup();
  inf >> palmt;
  inf >> jav;
  inf >> ntemp;
  for (i = 0; i < ntemp; i++) { inf >> tv; temp.push_back(tv); }
  temp.sortElimDup();
  inf >> sigma >> wt1 >> wt2 >> eps;
  redmass = wt1 * wt2 / (wt1 + wt2);
  inf >> ioptht >> ioptpr;
  inf >> ngrain;
  for (i = 0; i < ngrain; i++) { inf >> rhv; rho.push_back(rhv); }
  for (i = 0; i < ntemp; i++) { inf >> tv >> corrat; }
  inf >> nreacg;
  for (ich = 0; ich < nchan; ich++) {
    rtvl.clear();
    for (i = 0; i < nreacg; i++) { inf >> rtv; rtvl.push_back(rtv); }
    rate.push_back(rtvl);
  }
  nnonrg = ngrain - nreacg;
  upbound = ngrain;
  err3 *= 1e-3;
  inf.close();
  return NORM;
}

void well::printParETP(ostream &os, int vb) {       // --- printParETP ---
  if (vb > 1) {
    os << "well-" << index << ": thRhoGrad = " << thRhoGrad;
    if (fabs(thRhoGrad - DEF_thRhoGrad) < 1.e-6) { os << " (default)"; }
    os << endl;
    os << "well-" << index << ": redExpon = ";
    if (redExpFix < 0.) { os << "auto (default)\n"; }
    else { os << redExpFix << endl; }
  }
  os << "well-" << index << ": err3 = " << err3 << endl;
}

void well::printParCA(ostream &os) {                 // --- printParCA ---
  if (caWell)
    { os << "well-" << index << ": recombChan = " << actchan + 1 << endl; }
  os << "well-" << index << ": truncate = " << lowest;
  if (lowest == 0) { os << " (no truncation)"; }
  os << endl;
}

int well::setCurrentTp(double Tin, double pin) {   // --- setCurrentTp ---
  double densM, vMean, crossSec, Omega22;
  if (sigma < 1.e-3) { cout << "well: data not read.\n"; return ERRE; }
  current.T = Tin; current.p = pin;
  current.alpha = alpha1000 * pow(current.T / 1000., alpTex);
  crossSec = M_PI * sigma * sigma * 1.e-16;            // cm2
  vMean = CVMEAN * sqrt(current.T / redmass);          // cm s-1
  densM = current.p / RTCMMLC / current.T;             // molecules cm-3
  Omega22 = 1. / (0.636 + 0.567 * log10(current.T / eps));
  current.collfreq = crossSec * vMean * densM * Omega22;
  return NORM;
}

/*------------------------------------------------------------------------
     List of Wells
------------------------------------------------------------------------*/

listWells::listWells() { clear(); }               // --- (constructor) ---

void listWells::clear() {                                 // --- clear ---
  totSize = 0; posReactWell = -1; topCut = 10;
  kEthInt = DEFAULT_kEthInt; kEthOut = DEFAULT_kEthOut;
  tableIndices.clear();
}

int listWells::isValidIndex(int idx) {             // --- isValidIndex ---
  if (tableIndices.find(idx) != tableIndices.end()) { return true; }
  return false;
}

int listWells::isUsedIndex(int idx)                 // --- isUsedIndex ---
  { return isValidIndex(idx); }

int listWells::autoIndex() {                          // --- autoIndex ---
  int idx = 0;
  do { idx++; } while (isUsedIndex(idx));
  return idx;
}

int listWells::getPos(int idx) {                         // --- getPos ---
  if (!isValidIndex(idx)) { return -1; }
  return tableIndices[idx];
}

int listWells::add(well &w) {                               // --- add ---
  int idx = w.index, pos = size();
  if (isUsedIndex(idx)) { return ERRE; }
  push_back(w);
  tableIndices.insert(make_pair(idx, pos));
  return NORM;
}

int listWells::readUnimol() {                        // --- readUnimol ---
  iterator p;
  for (p = begin(); p < end(); p++)
    { if (p->readFromUnimol() == ERRE) { return ERRE; } }
  if (checkConsistency() == ERRE) { return ERRE; }
  if (checkConnect() == ERRE) { return ERRE; }
  alignSize();
  if (checkEqkE() == ERRE) { return ERRE; }
  setChanInfo();
  checkKout();
  return NORM;
}

void listWells::mergedTlist(tempList &tl) {         // --- mergedTlist ---
  iterator p;
  tl.clear();
  for (p = begin(); p < end(); p++) { tl.addList(p->temp); }
}

void listWells::mergedPlist(pressList &pl) {        // --- mergedPlist ---
  iterator p;
  pl.clear();
  for (p = begin(); p < end(); p++) { pl.addList(p->press); }
}

void listWells::printParETP(ostream &os, int vb) {  // --- printParETP ---
  iterator p;
  for (p = begin(); p < end(); p++) { p->printParETP(os, vb); }
  cout << "kEthInt / kEthOut = " << kEthInt << " / " << kEthOut << endl;
}

void listWells::printParCA(ostream &os) {            // --- printParCA ---
  iterator p;
  for (p = begin(); p < end(); p++) { p->printParCA(os); }
}

int listWells::reduceSize(int maxs) {                // --- reduceSize ---
  int redWell;
  iterator p;
  if (totSize == 0) { alignSize(); }
  if (totSize <= maxs) { return NORM; }
  redWell = (totSize - maxs + 3) / size();
  absTop -= redWell;
  totSize = 0;
  for (p = begin(); p < end(); p++) {
    p->upbound = absTop - p->offset;
    totSize += p->upbound - p->lowest;
    if (p->upbound <= p->nnonrg) { return ERRE; }
  }
  return NORM;
}

void listWells::setPosBmat() {                       // --- setPosBmat ---
  int posB = 0;
  iterator p;
  for (p = begin(); p < end(); p++) {
    p->posBmat = posB;
    posB += p->upbound - p->lowest;
  }
}

void listWells::setErr3(double e) {                     // --- setErr3 ---
  iterator p;
  for (p = begin(); p < end(); p++) { p->err3 = e; }
}

void listWells::setAlpPar(double a1k, double aEx) {   // --- setAlpPar ---
  iterator p;
  for (p = begin(); p < end(); p++)
    { p->alpha1000 = a1k; p->alpTex = aEx; }
}

int listWells::checkCA() {                              // --- checkCA ---
  int ncaWell = 0;
  iterator p;
  for (p = begin(); p < end(); p++) {
    if (p->caWell) { ncaWell++; }
  }
  if (ncaWell == 0)
    { cout << "No recombination channel.\n"; return ERRE; }
  else if (ncaWell > 1)
    { cout << "Too many recombination channels.\n"; return ERRE; }
  return NORM;
}

int listWells::checkConsistency() {            // --- checkConsistency ---
  int posw, grsz0, ncaw = 0;
  double redm0;
  if ((*this)[0].caWell) { ncaw++; }
  grsz0 = (*this)[0].grsize;
  redm0 = (*this)[0].redmass;
  for (posw = 1; posw < size(); posw++) {
    if ((*this)[posw].caWell) { ncaw++; }
    if ((*this)[posw].grsize != grsz0)
      { cout << "Grain size differs between wells.\n"; return ERRE; }
    if (fabs((*this)[posw].redmass - redm0) > 1.e-6)
      { cout << "Masses differ between wells.\n"; return ERRE; }
  }
  if (ncaw > 1)
    { cout << "More than one wells have recombChan.\n"; return ERRE; }
  return NORM;
}

int listWells::checkConnect() {                    // --- checkConnect ---
  int pos, ic, toIdx, toPos, chan, asym;
  vector<vector<int> > ncnx(size(), vector<int>(size(), 0));

  for (pos = 0; pos < size(); pos++) {
    for (ic = 0; ic < (*this)[pos].connect.size(); ic++) {
      if (!isValidIndex(toIdx = (*this)[pos].connect[ic].toWellIdx)) {
        cout << "well-" << (*this)[pos].index << ": Invalid connect-toWellIdx "
             << toIdx << endl;
        return ERRE;
      }
      toPos = tableIndices[toIdx];
      ncnx[pos][toPos]++;
      chan = (*this)[pos].connect[ic].chan;
      if ((chan < 0) || (chan >= (*this)[pos].nchan)) {
        cout << "well-" << (*this)[pos].index << ": Invalid connect-chan "
             << chan + 1 << endl;
        return ERRE;
      }
    }
  }
  asym = false;
  for (pos = 0; pos < size(); pos++) {
    for (toPos = pos + 1; toPos < size(); toPos++) {
      if (ncnx[toPos][pos] != ncnx[pos][toPos]) { asym = true; break; }
    }
    if (asym) { break; }
  }
  if (asym) {
    cout << "Asymmetric connection matrix.\n";
    for (pos = 0; pos < size(); pos++) {
      for (toPos = 0; toPos < size(); toPos++)
        { cout << ' ' << ncnx[pos][toPos]; }
      cout << endl;
    }
    return ERRE;
  }
  return NORM;
}

void listWells::alignSize() {                         // --- alignSize ---
  int pos, minOffset = (*this)[0].offset, top;

  // - set minimum offset to 0
  for (pos = 1; pos < size(); pos++) {
    if ((*this)[pos].offset < minOffset)
      { minOffset = (*this)[pos].offset; }
  }
  for (pos = 0; pos < size(); pos++) { (*this)[pos].offset -= minOffset; }
  // - align upbound & count total size
  absTop = (*this)[0].offset + (*this)[0].ngrain;
  for (pos = 1; pos < size(); pos++) {
    top = (*this)[pos].offset + (*this)[pos].ngrain;
    if (top < absTop) { absTop = top; }
  }
  absTop -= topCut;  // - top cut to avoid asymmetry problem
  totSize = 0;
  for (pos = 0; pos < size(); pos++) {
    (*this)[pos].upbound = absTop - (*this)[pos].offset;
    totSize += (*this)[pos].upbound - (*this)[pos].lowest;
  }
  cout << "listWells::alignSize absTop = " << absTop << endl;
}

int listWells::checkEqkE() {                          // --- checkEqkE ---
  int pos, toPos, ic, ig, botGr, ighere, igto, irhere, irto, ich, igdpcww;
  double kforw, kback, rhohere, rhoto, diffpc, dpcww, maxdpc = 0., corf;
  vector<int> forwChan, backChan;

  if (kEthInt < DEFAULT_kEthInt) { kEthInt = DEFAULT_kEthInt; }
  for (pos = 0; pos < size(); pos++) {
    for (toPos = pos + 1; toPos < size(); toPos++) {
      forwChan.clear();
      for (ic = 0; ic < (*this)[pos].connect.size(); ic++) {
        if (getPos((*this)[pos].connect[ic].toWellIdx) == toPos)
          { forwChan.push_back((*this)[pos].connect[ic].chan); }
      }
      if (forwChan.empty()) { continue; }
      backChan.clear();
      for (ic = 0; ic < (*this)[toPos].connect.size(); ic++) {
        if (getPos((*this)[toPos].connect[ic].toWellIdx) == pos)
          { backChan.push_back((*this)[toPos].connect[ic].chan); }
      }
      botGr = (*this)[pos].offset + (*this)[pos].nnonrg;
      if (((*this)[toPos].offset + (*this)[toPos].nnonrg) < botGr)
        { botGr = (*this)[toPos].offset + (*this)[toPos].nnonrg; }
      if (((*this)[pos].offset + (*this)[pos].lowest) > botGr)
        { botGr = (*this)[pos].offset + (*this)[pos].lowest; }
      if (((*this)[toPos].offset + (*this)[toPos].lowest) > botGr)
        { botGr = (*this)[toPos].offset + (*this)[toPos].lowest; }
      dpcww = 0.; igdpcww = -1;
      for (ig = botGr; ig < absTop; ig++) {
        ighere = ig - (*this)[pos].offset;
        irhere = ighere - (*this)[pos].nnonrg;
        igto = ig - (*this)[toPos].offset;
        irto = igto - (*this)[toPos].nnonrg;
        if ((ighere < 0) || (igto < 0)) { continue; }
        kforw = 0.;
        if (irhere >= 0) {
          for (ich = 0; ich < forwChan.size(); ich++)
            { kforw += (*this)[pos].rate[forwChan[ich]][irhere]; }
        }
        kback = 0.;
        if (irto >= 0) {
          for (ich = 0; ich < backChan.size(); ich++)
            { kback += (*this)[toPos].rate[backChan[ich]][irto]; }
        }
        rhohere = (*this)[pos].rho[ighere];
        rhoto = (*this)[toPos].rho[igto];
        diffpc = 0.;
        if ((kback < kEthInt) || (kforw < kEthInt)) {
          for (ich = 0; ich < forwChan.size(); ich++)
            { (*this)[pos].rate[forwChan[ich]][irhere] = 0.; }
          for (ich = 0; ich < backChan.size(); ich++)
            { (*this)[toPos].rate[backChan[ich]][irto] = 0.; }
        } else {
          diffpc = fabs((kforw / kback) / (rhoto / rhohere) - 1.) * 100.;
          if (diffpc > 1.) {
            corf = sqrt((rhoto / rhohere) / (kforw / kback));
            for (ich = 0; ich < forwChan.size(); ich++)
              { (*this)[pos].rate[forwChan[ich]][irhere] *= corf; }
            for (ich = 0; ich < backChan.size(); ich++)
              { (*this)[toPos].rate[backChan[ich]][irto] /= corf; }
          }
        }
        if (diffpc > dpcww) { dpcww = diffpc; igdpcww = ig; }
      }
      if (dpcww > maxdpc) { maxdpc = dpcww; }
      cout << "max asymmetry [%] between well" << (*this)[pos].index
           << " and well" << (*this)[toPos].index << " = " << dpcww
           << " at grid #" << igdpcww << endl;
    }
  }
  if (maxdpc > 60.) {
    cout << "ERROR: large k(E) max asymmetry [%] = " << maxdpc << endl;
    return ERRE;
  }
  if (maxdpc > 10.)
    { cout << "WARNING: k(E) max asymmetry [%] = " << maxdpc << endl; }
  return NORM;
}

void listWells::setChanInfo() {                     // --- setChanInfo ---
  int posw, ch, icn;
  nTotCh = 0; nTotOCh = 0; nTotStbW = 0;
  for (posw = 0; posw < size(); posw++) {
    (*this)[posw].chInf.clear();
    for (ch = 0; ch < (*this)[posw].nchan; ch++)
      { (*this)[posw].chInf.push_back(-1); }
    for (icn = 0; icn < (*this)[posw].connect.size(); icn++) {
      (*this)[posw].chInf[(*this)[posw].connect[icn].chan]
       = this->getPos((*this)[posw].connect[icn].toWellIdx);
    }
    (*this)[posw].nOutCh = 0; (*this)[posw].nStab = 0;
    for (ch = 0; ch < (*this)[posw].nchan; ch++) {
      if ((*this)[posw].chInf[ch] == -1) { (*this)[posw].nOutCh++; }
    }
    if ((*this)[posw].lowest > 0) { (*this)[posw].nStab++; }
    nTotOCh += (*this)[posw].nOutCh;
    nTotCh += (*this)[posw].nchan;
    nTotStbW += (*this)[posw].nStab;
  }
}

void listWells::checkKout() {                         // --- checkKout ---
  int posw, ich, ir;
  if (kEthOut < DEFAULT_kEthOut) { kEthOut = DEFAULT_kEthOut; }
  for (posw = 0; posw < size(); posw++) {
    for (ich = 0; ich < (*this)[posw].nchan; ich++) {
      if ((*this)[posw].chInf[ich] < 0) {
        for (ir = 0; ir < (*this)[posw].rate[ich].size(); ir++) {
          if ((*this)[posw].rate[ich][ir] < kEthOut)
            { (*this)[posw].rate[ich][ir] = 0.; }
        }
      }
    }
  }
}

/*------------------------------------------------------------------------
     Unimolecular Problem Solver
------------------------------------------------------------------------*/

/*----- public member functions ----------------------------------------*/

umolProb::umolProb() { clear(); }                 // --- (constructor) ---

void umolProb::clear() {                                  // --- clear ---
  probtype = Eigen; eigensolver = ESNone; timeevcond = TEconst;
  HPL = false; RFlxOut = false;
  maxCycLinSs = 300; maxCycItrSs = 1000;
  verbose = 0;
  psiz = 0;
  sizeLSDSRW = 0.5;
  T = 0.; p = 0.;
}

int umolProb::initMW(listWells &lw) {                    // --- initMW ---
  if (lw.totSize > MXSIZ) {
    if (lw.reduceSize(MXSIZ) == ERRE) {
      cout << "umolProb::initMW too small size available.\n";
      return ERRE;
    }
    cout << "umolProb::initMW size reduced to " << lw.totSize << ".\n";
  }
  psiz = lw.totSize;
  cout << "umolProb::initMW psiz = " << psiz << endl;
  lw.setPosBmat();
  return NORM;
}

void umolProb::calcHPLMW(listWells &lw, double Tin) { // --- calcHPLMW ---
  int posw;
  double tgnorm;
  T = Tin; p = -1.;
  setBoltzRflxMWall(lw, Tin);
  for (posw = 0; posw < lw.size(); posw++)
    { setSmat(lw[posw], lw[posw].posBmat); }
  applyStoV();
  tgnorm = normGMW(lw, v);
  integkEgMW(lw, v, tgnorm);
}

                                              // --- setBoltzRflxMWall ---
void umolProb::setBoltzRflxMWall(listWells &lw, double Tin) {
  int posw, shift, i;
  double gamma = exp(-lw[0].grsize / KB_WNK / Tin), tnorm = 0.;
  for (posw = 0; posw < lw.size(); posw++) {
    shift = lw[posw].posBmat - lw[posw].lowest;
    for (i = lw[posw].lowest; i < lw[posw].upbound; i++) {
      tnorm +=
       (v[i + shift] = lw[posw].rho[i] * pow(gamma, i + lw[posw].offset));
    }
  }
  for (i = 0; i < psiz; i++) { v[i] /= tnorm; }
}

                                              // --- setBoltzRflxMWsel ---
void umolProb::setBoltzRflxMWsel(listWells &lw, double Tin) {
  int poswr = lw.posReactWell;
  setBoltzRflx1W(lw[poswr], Tin, lw[poswr].posBmat);
}

                                                 // --- setBoltzRflx1W ---
void umolProb::setBoltzRflx1W(well &w, double Tin, int pos0) {
  int i, shift = pos0 - w.lowest;
  double gamma = exp(-w.grsize / KB_WNK / Tin), norm = 0.;
  for (i = w.lowest; i < w.upbound; i++)
    { norm += (v[i + shift] = w.rho[i] * pow(gamma, i)); }
  for (i = w.lowest; i < w.upbound; i++)
    { v[i + shift] /= norm; }
}

                                               // --- driveLinSsMWposv ---
int umolProb::driveLinSsMWposv(listWells &lw, double Tin, double pin) {
  int lcnt = 1, dcnt = 1, poswr = lw.posReactWell, posB, converged = false;
  if (poswr >= 0) { posB = lw[poswr].posBmat; }
  double knew, kold = 0., dknew, dkold = 0., step = 0.5;

  if (setMatMWinvSgn(lw, Tin, pin) == ERRE) { return ERRE; }
  if (solveLinEqMWposv(lw) == ERRE) { return ERRE; }
  do {
    knew = lw.koutTot; dknew = knew - kold;
    // if (knew < 0.) { step *= 0.1; dcnt = 0; }
    if (dcnt > 3) {
      dcnt = 0;
      if (dknew * dkold < 0) { step *= 0.5; }
      else {
        step *= 1.5;
        if (step > 1.) { step = 1.; }
      }
    }
    if (poswr < 0) { storeRflxAll(step); }
    else { storeRflx(lw[poswr], posB, step); }
    cout << "umolProb::driveLinSsMWposv ===== LOOP " << lcnt << " ====="
         << " step = " << step
         << ", k = " << knew << endl;
    if (lcnt >= maxCycLinSs) {
      cout << "umolProb::driveLinSsMWposv *** MAXCYCLE ***\n";
      return ITRERR;
    }
    if (fabs((kold - knew) / knew) <= 1.e-5) { converged = true; continue; }
    kold = knew; dkold = dknew;
    // - restore r[] only
    clearRflx();
    if (poswr < 0) { restoreSSG(); }
    else { restoreRflx(lw[poswr], posB); }
    applyStoR();
    if (solveLinEqMWposv2nd(lw) == ERRE) { return ERRE; }
    lcnt++; dcnt++;
  } while (!converged);
  return NORM;
}

                                                       // --- setMatMW ---
int umolProb::setMatMW(listWells &lw, double Tin, double pin) {
  int noffd, pwell, posB;
  T = Tin; p = pin;
  noffdiag = 0;
  if ((probtype == LinEq) || (probtype == LinSS) || (probtype == TimEv) ||
      (probtype == TimDs))
    { clearRflx(); }
  if ((probtype == TimEv) || (probtype == TimDs)) { clearV(); }
  for (pwell = 0; pwell < lw.size(); pwell++) {
    posB = lw[pwell].posBmat;
    if (drvSetETP(lw[pwell], posB, noffd) == ERRE) { return ERRE; }
    if (noffd > noffdiag) { noffdiag = noffd; }
    setRateSmat(lw[pwell], posB);
    if ((probtype == LinEq) && lw[pwell].caWell)
//      { setRflxChan(lw[pwell], posB); }
      { setRflxCPos(lw[pwell], posB); }
    if ((probtype == LinSS) && lw[pwell].isReactant)
      { restoreRflx(lw[pwell], posB); }
    if ((probtype == TimEv) && lw[pwell].caWell) {
      if (timeevcond == TEconst) { setRflxCPos(lw[pwell], posB); }
      else if (timeevcond == TEinit) { setVflxCPos(lw[pwell], posB); }
    }
  }
  if ((probtype == ItrSS) || ((probtype == LinSS) && (lw.posReactWell < 0)))
    { restoreSSG(); }
  setCrossRate(lw);
  if (symmetMat() == ERRE) { return ERRE; }
  for (pwell = 0; pwell < lw.size(); pwell++) {
    posB = lw[pwell].posBmat;
    makeMatBanded(lw[pwell], posB);
  }
  return NORM;
}

                                                 // --- setMatMWinvSgn ---
int umolProb::setMatMWinvSgn(listWells &lw, double Tin, double pin) {
  if (setMatMW(lw, Tin, pin) == ERRE) { return ERRE; }
  invSignB();
  return NORM;
}

                                             // --- solveEigenMWdsyevr ---
int umolProb::solveEigenMWdsyevr(listWells &lw) {
  int info, goodEigen, posw, ik, posmin;
  double tgnorm;
  eigensolver = ESdsyevr;
  posmin = psiz * 0.8;
  if (verbose > 0) { cout << " Calling lapack_dsyevr ...\n"; }
  info = lapack_dsyevr('V', 'I', 'U', psiz, (double **)b, MXSIZ, 0., 0.,
   posmin, psiz, 0., esiz, v, (double **)z, MXSIZ, isuppz, work, WMSIZ,
   iwork, WISIZ);
  if (verbose > 0) {
    cout << "posmin = " << posmin << ", esiz = " << esiz << endl;
    cout << "lapack DSYEVR: ";
    cout << "optimal WMSIZ = " << work[0];
    cout << ", optimal WISIZ = " << iwork[0] << endl;
  }
  if (info != 0) {
    cout << "umolProb::solveEigenMWdsyevr lapack_dsyevr returnd info = "
         << info << ".\n";
    return ERRE;
  }
  findLargeNegEigen();
  if (lw.posReactWell >= 0) {
    do {
      goodEigen = true;
      tgnorm = normGMW(lw, z[posLNE]);
      for (posw = 0; posw < lw.size(); posw++)
        { if (lw[posw].current.fpop < 0.) { goodEigen = false; } }
      if (lw[lw.posReactWell].current.fpop < 0.5) { goodEigen = false; }
      if (!goodEigen) {
        if (posLNE == 0) {
          cout << "umolProb::solveEigenMW no good eigen value was found.\n";
          return ERRE;
        } else {
          posLNE--;
        }
      }
    } while (!goodEigen);
  } else {
    tgnorm = normGMW(lw, z[posLNE]);
  }
  integkEgMW(lw, z[posLNE], tgnorm);
  return NORM;
}

                                               // --- solveLinEqMWposv ---
int umolProb::solveLinEqMWposv(listWells &lw) {
  int info;
  double tgnorm;
  if (verbose > 0) { cout << " Calling lapack_dposv ...\n"; }
  info = lapack_dposv('U', psiz, 1, (double **)b, MXSIZ, (double **)r,
                      MXSIZ);
  if (info != 0) {
    cout << "umolProb::solveLinEqMWposv lapack_dposv returnd info = " << info
         << ".\n";
    return ERRE;
  }
  tgnorm = normGMW(lw, r[0]);
  integkEgMW(lw, r[0], tgnorm);
  return NORM;
}

                                            // --- solveLinEqMWposv2nd ---
int umolProb::solveLinEqMWposv2nd(listWells &lw) {
  // - for the 2nd and later calls from driveLinSsMWposv
  int info;
  double tgnorm;
  if (verbose > 0) { cout << " Calling lapack_dpotrs ...\n"; }
  info = lapack_dpotrs('U', psiz, 1, (double **)b, MXSIZ, (double **)r,
                       MXSIZ);
  if (info != 0) {
    cout << "umolProb::solveLinEqMWposv2nd lapack_dpotrs returnd info = " << info
         << ".\n";
    return ERRE;
  }
  tgnorm = normGMW(lw, r[0]);
  integkEgMW(lw, r[0], tgnorm);
  return NORM;
}

                                                  // --- solveTimeEvMW ---
int umolProb::solveTimeEvMW(listWells &lw, timeList &tl) {
//  int mf = 21;
  int mf = 121;
  int iopt = 0, itask = 1, istate = 1, itol = 1;
//  int lrw = 22 + (9 + psiz) * psiz;
  int lrw, nnz;
  int liw = WISIZ;
  int itl;
  double rtol = 1.e-6, atol = 1.e-10, t = 0., tout;

  nnz = numNonZeroBelem();
  if (verbose > 0)
    { cout << " number of non-zero B-elements = " << nnz << endl; }
  if (nnz < psiz * psiz * sizeLSDSRW) { nnz = psiz * psiz * sizeLSDSRW; }
  lrw = 20 + 3 * nnz + 20 * psiz;
  double *rwork = new double[lrw];
  if (verbose > 0)
    { cout << " RWORK size = " << lrw << endl; }
  // - asuume tl[0] = 0.
  if (tl.size() > MXCOL) {
    cout << "umolProb::solveTimeEvMW too many time points.\n";
    delete[] rwork;
    return ERRE;
  }
  storeVtoZ(0);
  for (itl = 1; itl < tl.size(); itl++) {
    tout = tl[itl];
    if (verbose > 0) { cout << " Calling dlsodes ...\n"; }
//    cpp_dlsode(this, psiz, v, t, tout, itol, rtol, &atol, itask, istate,
//               iopt, rwork, lrw, iwork, liw, mf);
    cpp_dlsodes(this, psiz, v, t, tout, itol, rtol, &atol, itask, istate,
                iopt, rwork, lrw, iwork, liw, mf);
    if (istate < 0) {
      cout << "umolProb::solveTimeEvMW dlsodes returnd istate = " << istate
           << ".\n";
      delete[] rwork;
      return ERRE;
    }
    cout << " Time integration to t = " << t << " completed.\n";
    storeVtoZ(itl);
  }

  delete[] rwork;
  return NORM;
}

                                                  // --- solveTimeDsMW ---
int umolProb::solveTimeDsMW(listWells &lw, timeList &tl, double T0,
 listRateAW &lrat) {
  int mf = 121;
  int iopt = 0, itask = 1, istate = 1, itol = 1;
  int lrw, nnz;
  int liw = WISIZ;
  int itl;
  double rtol = 1.e-6, atol = 1.e-10, t = 0., tout, tgnorm;

  nnz = numNonZeroBelem();
  if (verbose > 0)
    { cout << " number of non-zero B-elements = " << nnz << endl; }
  if (nnz < psiz * psiz * sizeLSDSRW) { nnz = psiz * psiz * sizeLSDSRW; }
  lrw = 20 + 3 * nnz + 20 * psiz;
  double *rwork = new double[lrw];
  if (verbose > 0)
    { cout << " RWORK size = " << lrw << endl; }
  // - asuume tl[0] = 0.
  if (tl.size() > MXCOL) {
    cout << "umolProb::solveTimeDsMW too many time points.\n";
    delete[] rwork;
    return ERRE;
  }
  // - initial vector
  if (lw.posReactWell < 0) { setBoltzRflxMWall(lw, T0); }
  else { setBoltzRflxMWsel(lw, T0); }
  applyStoV();
  // - first point
  lrat.clear();
  tgnorm = normGMW(lw, v);
  integkEgMW(lw, v, tgnorm);
  storeVtoZ(0);
  lrat.push_back(getVrate(lw));

  for (itl = 1; itl < tl.size(); itl++) {
    tout = tl[itl];
    if (verbose > 0) { cout << " Calling dlsodes ...\n"; }
    cpp_dlsodes(this, psiz, v, t, tout, itol, rtol, &atol, itask, istate,
                iopt, rwork, lrw, iwork, liw, mf);
    if (istate < 0) {
      cout << "umolProb::solveTimeDsMW dlsodes returnd istate = " << istate
           << ".\n";
      delete[] rwork;
      return ERRE;
    }
    cout << " Time integration to t = " << t << " completed.\n";
    tgnorm = normGMW(lw, v);
    integkEgMW(lw, v, tgnorm);
    storeVtoZ(itl);
    lrat.push_back(getVrate(lw));
  }

  delete[] rwork;
  return NORM;
}

                                                           // --- lsdF ---
void umolProb::lsdF(int neq, double t, double *y, double *ydot) {
  // - calculate ydot (called by dlsodes)
  applyB(y, ydot); addVect(ydot, r[0]);
}

void umolProb::lsdsJAC(int neq, double t, double *y, int j, int *ian,
  // - calculate Jacobian (called by dlsodes)
 int *jan, double *pdj) {                               // --- lsdsJAC ---
  int ir;
  for (ir = 0; ir < psiz; ir++) {
    pdj[ir] = b[ir][j - 1];
  }
}

                                                 // --- outputVectorMW ---
void umolProb::outputVectorMW(ostream &os, listWells &lw) {
  int posw, i, iw, im, pos0;
  double *g, tgnorm, gsiz = lw[0].grsize;
  if (HPL) { g = v; }
  else if (probtype == Eigen) {
    if (eigensolver == ESdsyevr) { g = z[posLNE]; }
    else { g = b[posLNE]; }  // - for dsyev
  } else { g = r[0]; }
  tgnorm = normGMW(lw, g);
  os << "T[K],P[Torr]\n";
  if (RFlxOut) { os << T << ",-\n"; }
  else if (HPL) { os << T << ",inf\n"; }
  else { os << T << ',' << p << endl; }
  os << "grain#,E[cm-1]";
  for (posw = 0; posw < lw.size(); posw++) { os << ",g" << lw[posw].index; }
  os << endl;
  for (i = 0; i < lw.absTop; i++) {
    os << i << ',' << i * gsiz;
    for (posw = 0; posw < lw.size(); posw++) {
      os << ',';
      iw = i - lw[posw].offset;
      im = iw - lw[posw].lowest;
      pos0 = lw[posw].posBmat;
      if (iw >= 0) {
        if (im < 0) { os << '0'; }
        else { os << g[pos0 + im] * s[pos0 + im] / tgnorm; }
      }
    }
    os << endl;
  }
}

                                               // --- outputVecTimEvMW ---
void umolProb::outputVecTimEvMW(ostream &os, listWells &lw, timeList &tl) {
  // - output unnormalized vectors for TimEv problem
  int itl;
  int posw, i, iw, im, pos0;
  double *g, gsiz = lw[0].grsize;

  os << endl;
  os << "------------------------------------------------\n";
  os << "T[K],P[Torr]\n";
  os << T << ',' << p << endl;
  os << "------------------------------------------------\n";
  for (itl = 0; itl < tl.size(); itl++) {
    os << "\nt[s]," << tl[itl] << endl;
    os << "grain#,E[cm-1]";
    for (posw = 0; posw < lw.size(); posw++) { os << ",g" << lw[posw].index; }
    os << endl;
    g = z[itl];
    for (i = 0; i < lw.absTop; i++) {
      os << i << ',' << i * gsiz;
      for (posw = 0; posw < lw.size(); posw++) {
        os << ',';
        iw = i - lw[posw].offset;
        im = iw - lw[posw].lowest;
        pos0 = lw[posw].posBmat;
        if (iw >= 0) {
          if (im < 0) { os << '0'; }
          else { os << g[pos0 + im] * s[pos0 + im]; }
        }
      }
      os << endl;
    }
  }
}

                                                       // --- printMat ---
void umolProb::printMat(ostream &os, int start, double fact) {
  int i, j, end = start + 255;
  if (end > psiz) { end = psiz; }
  os << "umolProb::printMat\n";
  for (i = start; i < end; i++) {
    for (j = start; j < end; j++) { os << b[j][i] * fact << ','; }
    os << endl;
  }
}

void umolProb::outEigVal(ostream &os, int n) {        // --- outEigVal ---
  // - print out largest n eigenvalues
  int i, i0;
  i0 = esiz - n;
  if (i0 < 0) { i0 = 0; }
  for (i = i0; i < esiz; i++) {
    os << v[i];
    if (i < esiz - 1) { os << ','; }
    else { os << endl; }
  }
}

/*----- protected member functions -------------------------------------*/

                                                      // --- drvSetETP ---
int umolProb::drvSetETP(well &w, int pos0, int &noff) {
  int retSETP;
  if (w.setCurrentTp(T, p) == ERRE) { return ERRE; }
  if (w.redExpFix < 0.) {                       // --- auto
    w.current.redExpon = w.MIN_redExp;
    do {
      if ((retSETP = setETProb(w, pos0, noff)) == ERRE) {
        w.current.redExpon += w.STP_redExp;
        if (w.current.redExpon > w.MAX_redExp) {
          cout << "Exceeded MAX-redExpon. Abort.\n";
          break;
        } else {
          if (verbose > 1) {
            cout << "  Retrying with redExpon=" << w.current.redExpon << endl;
          }
        }
      }
    } while (retSETP == ERRE);
    if (retSETP == ERRE) { return ERRE; }
  } else {                                      // --- fixed
    w.current.redExpon = w.redExpFix;
    if (setETProb(w, pos0, noff) == ERRE) { return ERRE; }
  }
  return NORM;
}

                                                      // --- setETprob ---
int umolProb::setETProb(well &w, int pos0, int &noff) {
  int jc, ir, shift = pos0 - w.lowest, upref, ncutlow, ncut;
  double alpha, z, beta, gamma, Ke, pup, pdown, dnorm, usum, pupden;

  alpha = w.current.alpha; z = w.current.collfreq;
  beta = exp(-w.grsize / alpha);
  gamma = exp(-w.grsize / KB_WNK / T);
  upref = (int)(alpha * 1.5 / w.grsize) + 1;
  noff = (int)(log(w.err3) / log(beta)) + 1;
  ncutlow = 0;
  for (ir = w.ngrain - upref - 1; ir >= 0; ir--) {
    if (w.rho[ir + upref] / w.rho[ir] > w.thRhoGrad)
      { ncutlow = ir + 1; break; }
  }
  double *redfac = new double[ncutlow];
  for (ir = 0; ir < ncutlow; ir++) {
    redfac[ir] = pow(w.rho[ir + upref] / w.rho[ir], w.current.redExpon);
    if (redfac[ir] < 1.) { redfac[ir] = 1.; }
  }
// --- for debugging
//  cout << "umolProb::setETProb ncutlow=" << ncutlow << endl;
//
  double *norm = new double[w.ngrain];
  for (jc = w.ngrain - 1; jc >= w.lowest; jc--) {
    ncut = ncutlow;
    if (jc < ncutlow) { ncut = jc; }
    dnorm = (1. - pow(beta, jc - ncut + 1)) / (1. - beta);
    for (ir = ncut - 1; ir >=0; ir--)
      { dnorm += pow(beta, jc - ir) / redfac[ir]; }
    usum = 0.;
    pupden = 1.;
    if (jc < ncutlow) { pupden = redfac[jc]; }
    for (ir = jc + 1; ir < w.ngrain; ir++) {
      Ke = w.rho[ir] / w.rho[jc] * pow(gamma, ir - jc);
      pup = pow(beta, ir - jc) / pupden * Ke / norm[ir];
      usum += pup;
      if (ir < w.upbound) { b[jc + shift][ir + shift] = z * pup; }
    }
    if (usum >= 1.) {
      if (verbose > 1) {
      cout << "  umolProb::setETProb problem detected with thRhoGrad="
           << w.thRhoGrad << " and redExpon=" << w.current.redExpon << endl;
      }
      delete[] norm, redfac;
      return ERRE;
    }
    norm[jc] = dnorm / (1. - usum);
    for (ir = jc; ir >= w.lowest; ir--) {
      pdown = pow(beta, jc - ir) / norm[jc];
      if (ir < ncut) { pdown /= redfac[ir]; }
      if (ir == jc) { b[jc + shift][ir + shift] = z * (pdown - 1); }
      else { b[jc + shift][ir + shift] = z * pdown; }
    }
    // store sum of the downward transfer rate below w.lowest in d[]
    d[jc + shift] = 0.;
    for (ir = w.lowest - 1; ir >= 0; ir--) {
      pdown = pow(beta, jc - ir) / norm[jc];
      if (ir < ncut) { pdown /= redfac[ir]; }
      d[jc + shift] += z * pdown;
    }
// --- for debugging
//    cout << norm[jc] << ',' << usum << ',' << w.rho[jc] << endl;
//
  }
  delete[] norm, redfac;
  return NORM;
}

void umolProb::setRateSmat(well &w, int pos0) {     // --- setRateSmat ---
  setRate(w, pos0); setSmat(w, pos0);
}

void umolProb::setRate(well &w, int pos0) {             // --- setRate ---
  int i, irat, ch, shift = pos0 - w.lowest;
  double ktot;
  for (i = w.lowest; i < w.upbound; i++) {
    ktot = 0.;
    if ((irat = i - w.nnonrg) >= 0) {
      for (ch = 0; ch < w.nchan; ch++) { ktot += w.rate[ch][irat]; }
    }
    b[i + shift][i + shift] -= ktot;
  }
}

void umolProb::setSmat(well &w, int pos0) {             // --- setSmat ---
  int i, shift = pos0 - w.lowest;
  double gamma = exp(-w.grsize / KB_WNK / T), f;
  for (i = w.lowest; i < w.upbound; i++) {
    f = w.rho[i] * pow(gamma, i + w.offset);
    s[i + shift] = sqrt(f);
  }
}

void umolProb::setCrossRate(listWells &lw) {       // --- setCrossRate ---
  int pfr, pto, mfr, mto, mfr0, mto0, mfrL, mtoL, icn;
  int gfr, gto, rfr, rto;
  double kf, kb;
  vector<int> fwChan, bkChan;
  for (pfr = 0; pfr < lw.size(); pfr++) {
    for (pto = pfr + 1; pto < lw.size(); pto++) {
      mfr0 = lw[pfr].posBmat; mto0 = lw[pto].posBmat;
      mfrL = mfr0 + lw[pfr].upbound - lw[pfr].lowest;
      mtoL = mto0 + lw[pto].upbound - lw[pto].lowest;
      for (mfr = mfr0; mfr < mfrL; mfr++) {
        for (mto = mto0; mto < mtoL; mto++)
          { b[mfr][mto] = 0.; b[mto][mfr] = 0.; }
      }
      fwChan.clear();
      for (icn = 0; icn < lw[pfr].connect.size(); icn++) {
        if (lw.getPos(lw[pfr].connect[icn].toWellIdx) == pto)
          { fwChan.push_back(lw[pfr].connect[icn].chan); }
      }
      if (fwChan.empty()) { continue; }
      bkChan.clear();
      for (icn = 0; icn < lw[pto].connect.size(); icn++) {
        if (lw.getPos(lw[pto].connect[icn].toWellIdx) == pfr)
          { bkChan.push_back(lw[pto].connect[icn].chan); }
      }
      for (gfr = lw[pfr].lowest; gfr < lw[pfr].upbound; gfr++) {
        rfr = gfr - lw[pfr].nnonrg;
        kf = 0.;
        if (rfr >= 0) {
          for (icn = 0; icn < fwChan.size(); icn++)
            { kf += lw[pfr].rate[fwChan[icn]][rfr]; }
        }
        gto = gfr + lw[pfr].offset - lw[pto].offset;
        rto = gto - lw[pto].nnonrg;
        kb = 0.;
        if (rto >= 0) {
          for (icn = 0; icn < bkChan.size(); icn++)
            { kb += lw[pto].rate[bkChan[icn]][rto]; }
        }
        if (gto >= lw[pto].lowest) {
          mfr = mfr0 + gfr - lw[pfr].lowest;
          mto = mto0 + gto - lw[pto].lowest;
          b[mfr][mto] = kf; b[mto][mfr] = kb;
        }
      }
    }
  }
}

void umolProb::clearRflx() {                          // --- clearRflx ---
  int i;
  for (i = 0; i < psiz; i++) { r[0][i] = 0.; }
}

void umolProb::clearV() {                                // --- clearV ---
  int i;
  for (i = 0; i < psiz; i++) { v[i] = 0.; }
}

void umolProb::setRflxCPos(well &w, int pos0) {     // --- setRflxCPos ---
  int i, shift = pos0 - w.lowest, irat;
  double rf, sumrf = 0., gamma = exp(-w.grsize / KB_WNK / T);
  for (i = w.lowest; i < w.upbound; i++) {
    irat = i - w.nnonrg;
    if (irat < 0) { r[0][i + shift] = 0.; }
    else {
      rf = w.rho[i] * pow(gamma, i) * w.rate[w.actchan][irat];
      r[0][i + shift] = rf; sumrf += rf;
    }
  }
  for (i = w.lowest; i < w.upbound; i++) { r[0][i + shift] /= sumrf; }
}

void umolProb::setVflxCPos(well &w, int pos0) {     // --- setVflxCPos ---
  int i, shift = pos0 - w.lowest, irat;
  double rf, sumrf = 0., gamma = exp(-w.grsize / KB_WNK / T);
  for (i = w.lowest; i < w.upbound; i++) {
    irat = i - w.nnonrg;
    if (irat < 0) { v[i + shift] = 0.; }
    else {
      rf = w.rho[i] * pow(gamma, i) * w.rate[w.actchan][irat];
      v[i + shift] = rf; sumrf += rf;
    }
  }
  for (i = w.lowest; i < w.upbound; i++) { v[i + shift] /= sumrf; }
}

void umolProb::restoreRflx(well &w, int pos0) {     // --- restoreRflx ---
  int i, shift = pos0 - w.lowest;
  for (i = w.lowest; i < w.upbound; i++) { r[0][i + shift] = v[i + shift]; }
}

                                                      // --- storeRflx ---
void umolProb::storeRflx(well &w, int pos0, double stp) {
  int i, shift = pos0 - w.lowest;
  double norm = 0.;
  for (i = w.lowest; i < w.upbound; i++)
    { norm += r[0][i + shift] * s[i + shift]; }
  for (i = w.lowest; i < w.upbound; i++) {
    v[i + shift] *= (1. - stp);
    v[i + shift] += (r[0][i + shift] * s[i + shift] / norm) * stp;
  }
}

void umolProb::storeRflxAll(double stp) {          // --- storeRflxAll ---
  int i;
  double norm = 0.;
  for (i = 0; i < psiz; i++) { norm += r[0][i] * s[i]; }
  for (i = 0; i < psiz; i++) {
    v[i] *= (1. - stp);
    v[i] += (r[0][i] * s[i] / norm * stp);
  }
}

void umolProb::restoreSSG() {                        // --- restoreSSG ---
  int i;
  for (i = 0; i < psiz; i++) { r[0][i] = v[i]; }
}

void umolProb::storeSSG() {                            // --- storeSSG ---
  int i;
  double norm = 0.;
  for (i = 0; i < psiz; i++) {
    norm += (v[i] = r[0][i] * s[i]);
  }
  for (i = 0; i < psiz; i++) { v[i] /= norm; }
}

void umolProb::storeVtoZ(int pos) {                   // --- storeVtoZ ---
  int i;
  for (i = 0; i < psiz; i++) { z[pos][i] = v[i]; }
}

ratesAllWells &umolProb::getVrate(listWells &lw) {    // --- get rates ---
  int posw;
  static ratesAllWells rall;

  rall.clear();
  for (posw = 0; posw < lw.size(); posw++)
    { rall.push_back(lw[posw].current); }
  rall.koutTot = lw.koutTot;
  return rall;
}

int umolProb::symmetMat() {                           // --- symmetMat ---
  int ir, jc;
  double diffpc, maxdpc = 0.;
  for (ir = 0; ir < psiz; ir++) {
    for (jc = 0; jc < psiz; jc++) { b[jc][ir] *= s[jc] / s[ir]; }
  }
  for (ir = 0; ir < psiz - 1; ir++) {
    for (jc = ir + 1; jc < psiz; jc++) {
      if (b[jc][ir] < 0.)
        { cout << "Negative off-diagonal: [" << ir << "][" << jc << "]\n"; }
      else if (b[ir][jc] < 0.)
        { cout << "Negative off-diagonal: [" << ir << "][" << jc << "]\n"; }
      else if (b[jc][ir] * b[ir][jc] > 1.e-64) {
        diffpc = fabs(b[ir][jc] / b[jc][ir] - 1.) * 100.;
        if (diffpc > maxdpc) { maxdpc = diffpc; }
      }
    }
  }
  if (maxdpc > 5.) {
    cout << "ERROR: asymmetric b-mat - asymmetry [%] = " << maxdpc << endl;
    return ERRE;
  }
  if (maxdpc > 1.) {
    cout << "WARNING: b-mat max asymmetry [%] = " << maxdpc << endl;
  } else {
    if (verbose > 0)
      { cout << "b-mat max asymmetry [%] = " << maxdpc << endl; }
  }
  if ((probtype == LinEq) || (probtype == LinSS) || (probtype == ItrSS)
      || (probtype == TimEv))
    // also apply S to r
    { applyStoR(); }
  if (probtype == TimEv) { applyStoV(); }
  return NORM;
}

void umolProb::applyStoR() {                          // --- applyStoR ---
  int i;
  for (i = 0; i < psiz; i++) { r[0][i] /= s[i]; }
}

void umolProb::applyStoV() {                          // --- applyStoV ---
  int i;
  for (i = 0; i < psiz; i++) { v[i] /= s[i]; }
}

void umolProb::applyB(double *oprd, double *rslt) {      // --- applyB ---
  int irow, icol;
  for (irow = 0; irow < psiz; irow++) {
    rslt[irow] = 0.;
    for (icol = 0; icol < psiz; icol++)
      { rslt[irow] += b[icol][irow] * oprd[icol]; }
  }
}

void umolProb::addVect(double *tgt, double *adv) {      // --- addVect ---
  int i;
  for (i = 0; i < psiz; i++) { tgt[i] += adv[i]; }
}

int umolProb::numNonZeroBelem() {               // --- numNonZeroBelem ---
  int ir, ic, nnz = 0;
  for (ir = 0; ir < psiz; ir++) {
    for (ic = 0; ic < psiz; ic++) {
      if (b[ir][ic] != 0.) { nnz++; }
    }
  }
  return nnz;
}

void umolProb::makeMatBanded(well &w, int pos0) { // --- makeMatBanded ---
  int ir, ic, shift = pos0 - w.lowest;
  for (ir = w.lowest; ir < w.upbound; ir++) {
    for (ic = ir + noffdiag + 1; ic < w.upbound; ic++) {
      b[ic + shift][ir + shift] = 0.;
      b[ir + shift][ic + shift] = 0.;
    }
  }
}

void umolProb::invSignB() {                            // --- invSignB ---
  int ir, ic;
  for (ir = 0; ir < psiz; ir++) {
    for (ic = 0; ic < psiz; ic++) { b[ir][ic] = -b[ir][ic]; }
  }
}

void umolProb::findLargeNegEigen() {          // --- findLargeNegEigen ---
  int i;
  posLNE = esiz - 1;
  for (i = esiz - 1; i >= 0; i--) {
    if (v[i] < 0.) { posLNE = i; break; }
  }
}

                                                        // --- normG1W ---
double umolProb::normG1W(well &w, int pos0, double *g) {
  int i, imat;
  double sumg = 0.;
  for (i = w.lowest; i < w.upbound; i++) {
    imat = pos0 + i - w.lowest;
    sumg += g[imat] * s[imat];
  }
  return sumg;
}

double umolProb::normGMW(listWells &lw, double *g) {    // --- normGMW ---
  int posw, pos0;
  double totsumg = 0.;
  for (posw = 0; posw < lw.size(); posw++) {
    pos0 = lw[posw].posBmat;
    totsumg += (lw[posw].current.fpop = normG1W(lw[posw], pos0, g));
  }
  for (posw = 0; posw < lw.size(); posw++)
    { lw[posw].current.fpop /= totsumg; }
  return totsumg;
}

                                                     // --- integkEg1W ---
void umolProb::integkEg1W(well &w, int pos0, double *g, double gnorm) {
  int i, ch, shift = pos0 - w.lowest, irat, imat;
  double sumdg;
  vector<double> sumkg;

  if (probtype == Eigen) { w.current.keigen = -v[posLNE]; }
  sumdg = 0.; sumkg.clear();
  for (ch = 0; ch < w.nchan; ch++) { sumkg.push_back(0.); }
  for (i = w.lowest; i < w.upbound; i++) {
    imat = i + shift; irat = i - w.nnonrg;
    if (irat >= 0) {
      for (ch = 0; ch < w.nchan; ch++)
        { sumkg[ch] += g[imat] * s[imat] * w.rate[ch][irat]; }
    }
    sumdg += g[imat] * s[imat] * d[imat];
  }
  w.current.ktot = 0.;
  w.current.k.clear();
  for (ch = 0; ch < w.nchan; ch++) {
    w.current.k.push_back(sumkg[ch] / gnorm);
    w.current.ktot += sumkg[ch] / gnorm;
  }
  w.current.kstab = sumdg / gnorm;
  w.current.ktot += sumdg / gnorm;
  w.current.f.clear();
  for (ch = 0; ch < w.nchan; ch++) {
    w.current.f.push_back(w.current.k[ch] / w.current.ktot);
  }
  w.current.fstab = w.current.kstab / w.current.ktot;
}

                                                     // --- integkEgMW ---
void umolProb::integkEgMW(listWells &lw, double *g, double tgnorm) {
  int posw, pos0, ch;
  if (probtype == Eigen) { lw.keigen = -v[posLNE]; }
  lw.koutTot = 0.;
  for (posw = 0; posw < lw.size(); posw++) {
    pos0 = lw[posw].posBmat;
    integkEg1W(lw[posw], pos0, g, tgnorm);
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1) { lw.koutTot += lw[posw].current.k[ch]; }
    }
    if ((probtype == LinEq) || (probtype == LinSS))
      { lw.koutTot += lw[posw].current.kstab; }
  }
}

