/*========================================================================
     "ssulibb" - basic library functions
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2016 by A. Miyoshi, Univ. Tokyo
========================================================================*/

#ifndef _SSULIBB_H_
#define _SSULIBB_H_

#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include <string>
#include <vector>
#include <set>
#include <map>
#include <algorithm>
#include <cctype>
#include <cstring>
#include <cstdlib>
#include <cmath>
#include <cfloat>
using namespace std;

#ifndef true
#define true (-1)
#endif
#ifndef false
#define false 0
#endif
#ifndef M_PI
#define	M_PI    3.14159265358979323846
#endif
#ifndef NORM
#define NORM    0
#endif
#ifndef ERRE
#define ERRE    1
#endif
#define ITRERR  2
#define REVSTR  "SSUMES rev. 2014.05.20m2"
#define MAXLN1L 256
#define WAVNHRT 219474.6314        // cm-1 hartree-1
#define KJMLHRT 2625.499617        // kJ/mol hartree-1
#define KCAMHRT 627.5094687        // kcal/mol hartree-1
#define BIWAA2  16.8576290         // cm-1 amu A2
#define GHZWAVN 29.9792458         // GHz (cm-1)-1
#define KELWAVN 1.438775242        // K (cm-1)-1
#define JMLWAVN 11.96265646        // J mol-1 (cm-1)-1
#define CAMWAVN 2.859143514        // cal mol-1 (cm-1)-1
#define STDTMPR 298.15             // standard temperature [K] = 25 deg-C
#define RGASC   8.314472           // R (gas constant) in J K-1 mol-1
#define RBCMMLC 1.38065032e-22     //  in bar K-1 cm3 molecule-1
#define RTCMMLC 1.03557290E-19     //  in Torr K-1 cm3 molecule-1
#define KB_WNK  0.695035591        // kB in cm-1 K-1
#define CVMEAN  14550.8175         // vMean [cm s-1] coef.
#define CTRANS  1.87932128e+20
#define CSTR1B  (-1.151704861)
#define TSTK_H  2.08366428e+10     // kB/h

/*========================================================================
     General Purpose Funtions
========================================================================*/

void showRevision();
string getBasePart(string flname);
string getExtPart(string flname);
int getLineRmCmm(istream &inStrm, string &sLine, const string &cmmSyms);
int getLine(istream &inStrm, string &sLine);
int getLineTrim(istream &inStrm, string &sLine);
void strAllToUpper(string &s);
void strAllToLower(string &s);
string ltrim(string s);
string rtrim(string s);
string trim(string s);
vector<string> splitTokens(string &s, const string &spr);
string format(int v, int len, char filc);
string strListIdx(vector<int> &lst);
string strBoolVal(int &vBln);
int valBoolStr(string s);

/*========================================================================
     Common Library Functions
========================================================================*/

// --- Parameter File Input Functions ------------------------------------

int getLinePar(istream &inStrm, string &sLine);
int getlParMatch(istream &inStrm, const string &ref);
int getlParToks(istream &inStrm, vector<string> &toks, int &ntoks,
 string &sLine);
int getlParToksKey(istream &inStrm, string &key, vector<string> &toks,
 int &ntoks, string &sLine);

/*========================================================================
     Common Library Classes
========================================================================*/

/*------------------------------------------------------------------------
     List of Temperatures
------------------------------------------------------------------------*/

class tempList : public vector<double> {
 protected:                                    // --- data members ---
  enum errcode { INVkey, INVstep, INVrange, INVparN, INVtemp };
 public:                                   // --- member functions ---
  tempList();
  void setDefault();
  int addFrom(string slin);
  void sortElimDup();
  int addRange(double st, double et, double dt);
  int addRangeVS(vector<string> &vs);
  int addRecipRange(double numr, double stt, double lst, double stp);
  int addRecipRangeVS(vector<string> &vs);
  int addGauChebGrd(double Tmin, double Tmax, int nT);
  int addGauChebGrdVS(vector<string> &vs);
  int addList(vector<double> &vd);
  int addListVS(vector<string> &vs);
  ostream &print(ostream &os);
  double maxTemp();
 protected:
  int errms(errcode erc, string sv = "");
};                                                  // --- globals ---
ostream &operator<<(ostream &os, tempList &tl);

/*------------------------------------------------------------------------
     List of Pressures
------------------------------------------------------------------------*/

// internal values are in Torr unit

#define ATMTOR  760.               // atm -> Torr
#define PATOR   0.007500616827     // Pa  -> Torr
#define KPATOR  7.500616827        // kPa -> Torr
#define MPATOR  7500.616827        // MPa -> Torr
#define BARTOR  750.0616827        // bar -> Torr

class pressList : public vector<double> {
 public:                                       // --- data members ---
  string unit;
  double convFact;
 protected:
  enum errcode { INVkey, INVstep, INVrange, INVparN, INVpress, INVunit };
 public:                                   // --- member functions ---
  pressList();
  void setDefault();
  int setUnit(string ui);
  int addFrom(string slin);
  void sortElimDup();
  int addRange(double sp, double ep, double dp);
  int addRangeVS(vector<string> &vs);
  int addLog10Range(double stt, double lst, double stp);
  int addLog10RangeVS(vector<string> &vs);
  int addGauChebGrd(double Pmin, double Pmax, int nP);
  int addGauChebGrdVS(vector<string> &vs);
  int addList(vector<double> &vd);
  int addListVS(vector<string> &vs);
  ostream &print(ostream &os);
  double maxPress();
 protected:
  int errms(errcode erc, string sv = "");
};                                                  // --- globals ---
ostream &operator<<(ostream &os, pressList &pl);

/*------------------------------------------------------------------------
     List of Times
------------------------------------------------------------------------*/

class timeList : public vector<double> {
 public:                                       // --- data members ---
  double timeBase;
 protected:
  enum errcode { INVkey, INVstep, INVrange, INVparN, INVtime };
 public:                                   // --- member functions ---
  timeList();
  void setDefault();
  int setBase(double tb);
  int addFrom(string slin);
  void sortElimDup();
  int addRange(double st, double et, double dt);
  int addRangeVS(vector<string> &vs);
  int addLog10Range(double stt, double lst, double stp);
  int addLog10RangeVS(vector<string> &vs);
  int addList(vector<double> &vd);
  int addListVS(vector<string> &vs);
  void addZero();
  ostream &print(ostream &os);
  double maxTime();
 protected:
  int errms(errcode erc, string sv = "");
};                                                  // --- globals ---
ostream &operator<<(ostream &os, timeList &tl);

#endif

