/*========================================================================
     "ssulibb" - basic library functions
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2014 by A. Miyoshi, Univ. Tokyo
                                  created: Sept. 22, 2009 (from gpoplibf)
                                   edited: Jan.  25, 2010
                                   edited: June   6, 2011
                                   edited: May   20, 2014
                              last edited: Sept.  2, 2016
========================================================================*/

#include "ssulibb.h"

/*========================================================================
     General Purpose Funtions
========================================================================*/

/*..... show revision number ...........................................*/

void showRevision() { cout << REVSTR << endl; }

/*..... get base part from a filename ..................................*/

string getBasePart(string flname) {
  string basname;
  string::size_type p;
  if ((p = flname.rfind('.')) == string::npos) {
    basname = flname;
  } else {
    basname = flname.substr(0, p);
  }
  return basname;
}

/*..... get extension part from a filename .............................*/

string getExtPart(string flname) {
  string extname;
  string::size_type p;
  if ((p = flname.rfind('.')) == string::npos) {
    extname.erase();
  } else {
    extname = flname.substr(p + 1);
  }
  return extname;
}

/*..... get a line from stream with removal of comment part ............*/

int getLineRmCmm(istream &inStrm, string &sLine, const string &cmmSyms) {
  int ics;
  char cs;
  string::size_type p;
  do {
    if (getLine(inStrm, sLine) == ERRE) { return ERRE; }
    for (ics = 0; ics < cmmSyms.length(); ics++) {
      cs = cmmSyms[ics];
      if ((p = sLine.find(cs)) != string::npos)
        { sLine = sLine.substr(0, p); }
    }
    sLine = trim(sLine);
  } while (sLine == "");
  return NORM;
}

/*..... get a line from stream .........................................*/

int getLine(istream &inStrm, string &sLine) {
  static char inBuf[MAXLN1L];
  char c;
  int p = 0;
  sLine = "";
  if (inStrm.get(c).eof()) { return ERRE; }
  do {
    if (c == '\n') { break; }
    if (c != '\r') {
      inBuf[p] = c; p++;
      if (p == (MAXLN1L - 1)) { break; }
    }
  } while (!inStrm.get(c).eof());
  inBuf[p] = '\0';
  sLine = inBuf;
  return NORM;
}

/*..... get a line from stream and trim spaces .........................*/

int getLineTrim(istream &inStrm, string &sLine) {
  if (getLine(inStrm, sLine) == ERRE) { return ERRE; }
  sLine = trim(sLine);
  return NORM;
}

/*..... convert string to upper case ...................................*/

void strAllToUpper(string &s) {
  long ls = s.length(), p;
  for (p = 0; p < ls; p++) { s[p] = toupper(s[p]); }
}

/*..... convert string to lower case ...................................*/

void strAllToLower(string &s) {
  long ls = s.length(), p;
  for (p = 0; p < ls; p++) { s[p] = tolower(s[p]); }
}

/*..... trim preceding & succeeding spaces in the string ...............*/

string ltrim(string s) {
  long p, first = s.length();
  for (p = 0; p < s.length(); p++) {
    if ((s[p] != ' ') && (s[p] != '\t')) { first = p; break; }
  }
  return s.substr(first);
}

string rtrim(string s) {
  long p, last = -1;
  for (p = s.length() - 1; p >= 0; p--) {
    if ((s[p] != ' ') && (s[p] != '\t')) { last = p; break; }
  }
  return s.substr(0, last + 1);
}

string trim(string s) {
  string rs = s;
  rs = ltrim(rs); rs = rtrim(rs);
  return rs;
}

/*..... split string to tokens .........................................*/

vector<string> splitTokens(string &s, const string &spr) {
  long pos, lastpos = -1, slen = s.length();
  string tok;
  vector<string> rs;
  rs.clear();
  for (pos = 0; pos < slen; pos++) {
    if (spr.find(s[pos]) != string::npos) {
      if (lastpos + 1 < pos) {
        tok = s.substr(lastpos + 1, pos - lastpos - 1);
        rs.push_back(tok);
      }
      lastpos = pos;
    }
  }
  if (lastpos < slen - 1) {
    tok = s.substr(lastpos + 1);
    rs.push_back(tok);
  }
  return rs;
}

/*..... string formatting ..............................................*/

string format(int v, int len, char filc) {
  ostringstream os;
  os << setw(len) << setfill(filc) << v;
  return os.str();
}

/*..... prepare a string of list of indices ............................*/

string strListIdx(vector<int> &lst) {
  int i, n;
  ostringstream oss;
  n = lst.size();
  for (i = 0; i < n; i++) {
    if (i > 0) { oss << '-'; }
    oss << (lst[i] + 1);
  }
  return oss.str();
}

/*..... prepare a string of a boolean value ............................*/

string strBoolVal(int &vBln) {
  string rets;
  if (vBln) { rets = "true"; } else { rets = "false"; }
  return rets;
}

/*..... get a boolean value from a string ..............................*/

int valBoolStr(string s) {
  int retv;
  strAllToUpper(s);
  if (s == "TRUE") { retv = true; } else { retv = false; }
  return retv;
}

/*========================================================================
     Common Library Functions
========================================================================*/

// --- Parameter File Input Functions ------------------------------------

/*----- get a line from stream removing comment part (#!) --------------*/

int getLinePar(istream &inStrm, string &sLine) {
  return getLineRmCmm(inStrm, sLine, "#!");
}

/*----- get a line and check for matching ------------------------------*/

int getlParMatch(istream &inStrm, const string &ref) {
  string sLine;
  if (getLinePar(inStrm, sLine) == ERRE) { return ERRE; }
  if (sLine != ref) { return ERRE; }
  return NORM;
}

/*----- get a line and token-splitted strings --------------------------*/
int getlParToks(istream &inStrm, vector<string> &toks, int &ntoks,
 string &sLine) {
  if (getLinePar(inStrm, sLine) == ERRE) { return ERRE; }
  toks = splitTokens(sLine, " ,\t"); ntoks = toks.size();
  if (ntoks <= 0) { return ERRE; }
  return NORM;
}

/*----- get a line, token-splitted strings, and key --------------------*/

int getlParToksKey(istream &inStrm, string &key, vector<string> &toks,
 int &ntoks, string &sLine) {
  if (getlParToks(inStrm, toks, ntoks, sLine) == ERRE) { return ERRE; }
  key = toks[0];
  return NORM;
}

/*========================================================================
     Common Library Classes
========================================================================*/

/*------------------------------------------------------------------------
     List of Temperatures
------------------------------------------------------------------------*/

tempList::tempList() { clear(); }                 // --- (constructor) ---

void tempList::setDefault()                          // --- setDefault ---
  { clear(); addRange(300., 1500., 100.); }

int tempList::addFrom(string slin) {   // --- add from an input string ---
  string key;
  vector<string> tkns;
  tkns = splitTokens(slin, " ,\t");
  if (tkns.empty()) { return errms(INVparN); }
  key = tkns[0];
  if (key == "tempRange") { return addRangeVS(tkns); }
  else if (key == "tempRecipRange") { return addRecipRangeVS(tkns); }
  else if (key == "tempGauChebGrd") { return addGauChebGrdVS(tkns); }
  else if (key == "tempList") { return addListVS(tkns); }
  return errms(INVkey, key);
}

                          // --- sort and eliminate duplicated element ---
void tempList::sortElimDup() {
  int id, nelm = size();
  double dlast;
  vector<double> vd = *this;
  if (nelm < 2) { return; }
  sort(vd.begin(), vd.end());
  clear(); push_back(vd[0]); dlast = vd[0];
  for (id = 1; id < nelm; id++) {
    if ((vd[id] - dlast) > 1.e-10) { push_back(vd[id]); dlast = vd[id]; }
  }
}

                                    // --- add a range of temperatures ---
int tempList::addRange(double st, double et, double dt) {
  double t;
  if (dt < 1.e-6) { return errms(INVstep); }
  if ((st < 1.e-6) || (et < st)) { return errms(INVrange); }
  if (((et - st) / dt) > 1.e4) { return errms(INVstep); }
  for (t = st; t < (et + dt * .1); t += dt) { push_back(t); }
  sortElimDup();
  return NORM;
}

                                       // --- add a range from strings ---
int tempList::addRangeVS(vector<string> &vs) {
  // - assumes that vs[0] is the key "tempRange"
  double st, et, dt;
  if (vs.size() < 4) { return errms(INVparN); }
  st = atof(vs[1].c_str()); et = atof(vs[2].c_str());
  dt = atof(vs[3].c_str());
  return addRange(st, et, dt);
}

                            // --- add a range from reciprocal numbers ---
int tempList::addRecipRange(double numr, double stt, double lst,
 double stp) {
  double rt;
  if (stp < 1.e-12) { return errms(INVstep); }
  if ((stt < 1.e-12) || (lst < stt)) { return errms(INVrange); }
  if (((lst - stt) / stp) > 1.e4) { return errms(INVstep); }
  for (rt = stt; rt < (lst + stp * .1); rt += stp)
   { push_back(numr / rt); }
  sortElimDup();
  return NORM;
}

                 // --- add a range from reciprocal numbers in strings ---
int tempList::addRecipRangeVS(vector<string> &vs) {
  // - assumes that vs[0] is the key "tempRecipRange"
  double numr, stt, lst, stp;
  if (vs.size() < 5) { return errms(INVparN); }
  numr = atof(vs[1].c_str()); stt = atof(vs[2].c_str());
  lst = atof(vs[3].c_str()); stp = atof(vs[4].c_str());
  return addRecipRange(numr, stt, lst, stp);
}

                                // --- add Gauss-Chebyshev grid points ---
int tempList::addGauChebGrd(double Tmin, double Tmax, int nT) {
  int i;
  double Ttil, Tinv, Tmini = 1. / Tmin, Tmaxi = 1. / Tmax;
  for (i = 0; i < nT; i++) {
    Ttil = cos((2. * i + 1.) / 2. / nT * M_PI);
    Tinv = (Ttil * (Tmaxi - Tmini) + (Tmaxi + Tmini)) / 2.;
    push_back(1. / Tinv);
  }
  sortElimDup();
  return NORM;
}

                     // --- add Gauss-Chebyshev grid points in strings ---
int tempList::addGauChebGrdVS(vector<string> &vs) {
  int nT;
  double Tmin, Tmax;
  if (vs.size() < 4) { return errms(INVparN); }
  Tmin = atof(vs[1].c_str()); Tmax = atof(vs[2].c_str());
  nT = atoi(vs[3].c_str());
  return addGauChebGrd(Tmin, Tmax, nT);
}

                                     // --- add a list of temperatures ---
int tempList::addList(vector<double> &vd) {
  int id, nd = vd.size();
  for (id = 0; id < nd; id++) {
    if (vd[id] < 1.e-6) { return errms(INVtemp); }
    push_back(vd[id]);
  }
  sortElimDup();
  return NORM;
}

                          // --- add a list of temperatures in strings ---
int tempList::addListVS(vector<string> &vs) {
  // - assumes that vs[0] is the key "tempList"
  int is, ns = vs.size();
  vector<double> vd;
  vd.clear();
  for (is = 1; is < ns; is++) { vd.push_back(atof(vs[is].c_str())); }
  if (!vd.empty()) { return addList(vd); }
  return NORM;
}

ostream &tempList::print(ostream &os) {                   // --- print ---
  int itl, ntl = size();
  os << "tempList";
  for (itl = 0; itl < ntl; itl++) { os << " " << (*this)[itl]; }
  os << endl;
  return os;
}

double tempList::maxTemp()                              // --- maxTemp ---
  { return *max_element(begin(), end()); }

int tempList::errms(errcode erc, string sv) {    // --- error messages ---
  string msg = "tempList: ";
  switch(erc) {
    case INVkey: msg += "Invalid key[" + sv + "]."; break;
    case INVstep: msg += "Invalid (too small) step input."; break;
    case INVrange: msg += "Invalid range input."; break;
    case INVparN: msg += "Too small number of input parameters."; break;
    case INVtemp: msg += "Invalid temperature (< 0) input."; break;
  }
  cout << msg << endl;
  return ERRE;
}

ostream &operator<<(ostream &os, tempList &tl)    //------ operator << ---
  { return tl.print(os); }

/*------------------------------------------------------------------------
     List of Pressures
------------------------------------------------------------------------*/

pressList::pressList()                            // --- (constructor) ---
  { clear(); setUnit("Torr"); }

void pressList::setDefault() {                       // --- setDefault ---
  clear();
  setUnit("atm"); addLog10Range(-5., 5., 1.); setUnit("Torr");
}

int pressList::setUnit(string ui) {                     // --- setUnit ---
  string u = ui;
  strAllToLower(u);
  if (u == "torr") { unit = "Torr"; convFact = 1.; }
  else if (u == "atm") { unit = "atm"; convFact = ATMTOR; }
  else if (u == "pa") { unit = "Pa"; convFact = PATOR; }
  else if (u == "kpa") { unit = "kPa"; convFact = KPATOR; }
  else if (u == "mpa") { unit = "MPa"; convFact = MPATOR; }
  else if (u == "bar") { unit = "bar"; convFact = BARTOR; }
  else { return errms(INVunit, ui); }
  return NORM;
}

int pressList::addFrom(string slin) {  // --- add from an input string ---
  string key;
  vector<string> tkns;
  tkns = splitTokens(slin, " ,\t");
  if (tkns.empty()) { return errms(INVparN); }
  key = tkns[0];
  if (key == "pressRange") { return addRangeVS(tkns); }
  else if (key == "pressLog10Range") { return addLog10RangeVS(tkns); }
  else if (key == "pressGauChebGrd") { return addGauChebGrdVS(tkns); }
  else if (key == "pressList") { return addListVS(tkns); }
  else if (key == "pressUnit") { return setUnit(tkns[1]); }
  return errms(INVkey, key);
}

                          // --- sort and eliminate duplicated element ---
void pressList::sortElimDup() {
  int id, nelm = size();
  double dlast;
  vector<double> vd = *this;
  if (nelm < 2) { return; }
  sort(vd.begin(), vd.end());
  clear(); push_back(vd[0]); dlast = vd[0];
  for (id = 1; id < nelm; id++) {
    if ((vd[id] - dlast) > 1.e-64) { push_back(vd[id]); dlast = vd[id]; }
  }
}

                                       // --- add a range of pressures ---
int pressList::addRange(double sp, double ep, double dp) {
  double p;
  if (dp < 1.e-64) { return errms(INVstep); }
  if ((sp < 1.e-64) || (ep < sp)) { return errms(INVrange); }
  if (((ep - sp) / dp) > 1.e4) { return errms(INVstep); }
  for (p = sp; p < (ep + dp * .1); p += dp)
    { push_back(p * convFact); }
  sortElimDup();
  return NORM;
}

                                       // --- add a range from strings ---
int pressList::addRangeVS(vector<string> &vs) {
  // - assumes that vs[0] is the key "pressRange"
  double sp, ep, dp;
  if (vs.size() < 4) { return errms(INVparN); }
  sp = atof(vs[1].c_str()); ep = atof(vs[2].c_str());
  dp = atof(vs[3].c_str());
  return addRange(sp, ep, dp);
}

                                        // --- add a logarithmic range ---
int pressList::addLog10Range(double stt, double lst, double stp) {
  double lp;
  if (stp < 1.e-12) { return errms(INVstep); }
  if ((stt < -100.) || (lst < stt)) { return errms(INVrange); }
  if (((lst - stt) / stp) > 1.e4) { return errms(INVstep); }
  for (lp = stt; lp < (lst + stp * .1); lp += stp)
   { push_back(pow(10., lp) * convFact); }
  sortElimDup();
  return NORM;
}

                             // --- add a logarithmic range in strings ---
int pressList::addLog10RangeVS(vector<string> &vs) {
  // - assumes that vs[0] is the key "pressLog10Range"
  double stt, lst, stp;
  if (vs.size() < 4) { return errms(INVparN); }
  stt = atof(vs[1].c_str()); lst = atof(vs[2].c_str());
  stp = atof(vs[3].c_str());
  return addLog10Range(stt, lst, stp);
}

                                // --- add Gauss-Chebyshev grid points ---
int pressList::addGauChebGrd(double Pmin, double Pmax, int nP) {
  int i;
  double Ptil, logP, logPmin = log(Pmin), logPmax = log(Pmax);
  for (i = 0; i < nP; i++) {
    Ptil = cos((2. * i + 1.) / 2. / nP * M_PI);
    logP = (Ptil * (logPmax - logPmin) + (logPmax + logPmin)) / 2.;
    push_back(exp(logP) * convFact);
  }
  sortElimDup();
  return NORM;
}

                     // --- add Gauss-Chebyshev grid points in strings ---
int pressList::addGauChebGrdVS(vector<string> &vs) {
  int nP;
  double Pmin, Pmax;
  if (vs.size() < 4) { return errms(INVparN); }
  Pmin = atof(vs[1].c_str()); Pmax = atof(vs[2].c_str());
  nP = atoi(vs[3].c_str());
  return addGauChebGrd(Pmin, Pmax, nP);
}

                                        // --- add a list of pressures ---
int pressList::addList(vector<double> &vd) {
  int id, nd = vd.size();
  for (id = 0; id < nd; id++) {
    if (vd[id] < 1.e-64) { return errms(INVpress); }
    push_back(vd[id] * convFact);
  }
  sortElimDup();
  return NORM;
}

                             // --- add a list of pressures in strings ---
int pressList::addListVS(vector<string> &vs) {
  // - assumes that vs[0] is the key "pressList"
  int is, ns = vs.size();
  vector<double> vd;
  vd.clear();
  for (is = 1; is < ns; is++) { vd.push_back(atof(vs[is].c_str())); }
  if (!vd.empty()) { return addList(vd); }
  return NORM;
}

ostream &pressList::print(ostream &os) {                  // --- print ---
  int ipl, npl = size();
  os << "pressList";
  for (ipl = 0; ipl < npl; ipl++) { os << " " << (*this)[ipl]; }
  os << endl;
  return os;
}

double pressList::maxPress()                           // --- maxPress ---
  { return *max_element(begin(), end()); }

int pressList::errms(errcode erc, string sv) {   // --- error messages ---
  string msg = "pressList: ";
  switch(erc) {
    case INVkey: msg += "Invalid key [" + sv + "]."; break;
    case INVstep: msg += "Invalid (too small) step input."; break;
    case INVrange: msg += "Invalid range input."; break;
    case INVparN: msg += "Too small number of input parameters."; break;
    case INVpress: msg += "Invalid pressure (< 0) input."; break;
    case INVunit: msg += "Invalid unit [" + sv + "]."; break;
  }
  cout << msg << endl;
  return ERRE;
}

ostream &operator<<(ostream &os, pressList &pl)   //------ operator << ---
  { return pl.print(os); }

/*------------------------------------------------------------------------
     List of Times
------------------------------------------------------------------------*/

timeList::timeList()                              // --- (constructor) ---
  { clear(); setBase(1.); }

void timeList::setDefault() {                        // --- setDefault ---
  clear(); setBase(1.);
  addLog10Range(-6.,-1.,1.);
}

int timeList::setBase(double tb) {                      // --- setBase ---
  if (tb < 1.e-64) { return errms(INVtime); }
  timeBase = tb;
  return NORM;
}

int timeList::addFrom(string slin) {                    // --- addFrom ---
  string key;
  vector<string> tkns;
  tkns = splitTokens(slin, " ,\t");
  if (tkns.empty()) { return errms(INVparN); }
  key = tkns[0];
  if (key == "timeRange") { return addRangeVS(tkns); }
  else if (key == "timeLog10Range") { return addLog10RangeVS(tkns); }
  else if (key == "timeList") { return addListVS(tkns); }
  else if (key == "timeBase") { return setBase(atof(tkns[1].c_str())); }
  return errms(INVkey, key);
}

void timeList::sortElimDup() {                      // --- sortElimDup ---
  int id, nelm = size();
  double dlast;
  vector<double> vd = *this;
  if (nelm < 2) { return; }
  sort(vd.begin(), vd.end());
  clear(); push_back(vd[0]); dlast = vd[0];
  for (id = 1; id < nelm; id++) {
    if ((vd[id] - dlast) > 1.e-64) { push_back(vd[id]); dlast = vd[id]; }
  }
}

                                                       // --- addRange ---
int timeList::addRange(double st, double et, double dt) {
  double t;
  if (dt < 1.e-64) { return errms(INVstep); }
  if ((st < 0.) || (et < st)) { return errms(INVrange); }
  if (((et - st) / dt) > 1.e4) { return errms(INVstep); }
  for (t = st; t < (et + dt * .1); t += dt)
    { push_back(t * timeBase); }
  sortElimDup();
  return NORM;
}

int timeList::addRangeVS(vector<string> &vs) {       // --- addRangeVS ---
  // - assumes that vs[0] is the key "timeRange"
  double st, et, dt;
  if (vs.size() < 4) { return errms(INVparN); }
  st = atof(vs[1].c_str()); et = atof(vs[2].c_str());
  dt = atof(vs[3].c_str());
  return addRange(st, et, dt);
}

                                                  // --- addLog10Range ---
int timeList::addLog10Range(double stt, double lst, double stp) {
  double lt;
  if (stp < 1.e-12) { return errms(INVstep); }
  if ((stt < -100.) || (lst < stt)) { return errms(INVrange); }
  if (((lst - stt) / stp) > 1.e4) { return errms(INVstep); }
  for (lt = stt; lt < (lst + stp * .1); lt += stp)
   { push_back(pow(10., lt) * timeBase); }
  sortElimDup();
  return NORM;
}

                                                // --- addLog10RangeVS ---
int timeList::addLog10RangeVS(vector<string> &vs) {
  // - assumes that vs[0] is the key "timeLog10Range"
  double stt, lst, stp;
  if (vs.size() < 4) { return errms(INVparN); }
  stt = atof(vs[1].c_str()); lst = atof(vs[2].c_str());
  stp = atof(vs[3].c_str());
  return addLog10Range(stt, lst, stp);
}

int timeList::addList(vector<double> &vd) {             // --- addList ---
  int id, nd = vd.size();
  for (id = 0; id < nd; id++) {
    if (vd[id] < 0.) { return errms(INVtime); }
    push_back(vd[id] * timeBase);
  }
  sortElimDup();
  return NORM;
}

int timeList::addListVS(vector<string> &vs) {         // --- addListVS ---
  // - assumes that vs[0] is the key "timeList"
  int is, ns = vs.size();
  vector<double> vd;
  vd.clear();
  for (is = 1; is < ns; is++) { vd.push_back(atof(vs[is].c_str())); }
  if (!vd.empty()) { return addList(vd); }
  return NORM;
}

void timeList::addZero() {                              // --- addZero ---
  push_back(0.);
  sortElimDup();
}

ostream &timeList::print(ostream &os) {                   // --- print ---
  int itl, ntl = size();
  os << "timeList";
  for (itl = 0; itl < ntl; itl++) { os << " " << (*this)[itl]; }
  os << endl;
  return os;
}

double timeList::maxTime()                              // --- maxTime ---
  { return *max_element(begin(), end()); }

int timeList::errms(errcode erc, string sv) {    // --- error messages ---
  string msg = "timeList: ";
  switch(erc) {
    case INVkey: msg += "Invalid key [" + sv + "]."; break;
    case INVstep: msg += "Invalid (too small) step input."; break;
    case INVrange: msg += "Invalid range input."; break;
    case INVparN: msg += "Too small number of input parameters."; break;
    case INVtime: msg += "Invalid time (< 0) input."; break;
  }
  cout << msg << endl;
  return ERRE;
}

ostream &operator<<(ostream &os, timeList &tl)       // --- operator<< ---
  { return tl.print(os); }

