/*========================================================================
     "kcheb" - calculates rate coefficients from Chebyshev polynomial
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2009 by A. Miyoshi, Univ. Tokyo
                                  created: Dec.  19, 2009
                              last edited: Dec.  21, 2009
--------------------------------------------------------------------------
 usage: kcheb basename(.ext)
--------------------------------------------------------------------------
 - If no extension is given, default extension '.inp' is assumed
 - output is written in [basename]_kcheb_out.csv
========================================================================*/

#include "ssulibb.h"
#include "numlibChb.h"

/*===== main function ==================================================*/

int main(int argc, char *argv[]) {
  int ntkns, nParT = -1, nParP = -1, it, ip, iTemp, iPress;
  double Tmin = -1., Tmax, Pmin = -1., Pmax, T, P, Patm, k;
  string inpfln, basfln, extfln, csvfln;
  string key, slin;
  vector<double> crow;
  vector<vector<double> > coef;
  vector<string> tkns;
  ifstream inf;
  ofstream cof;
  tempList tlst;
  pressList plst;
  chebyRate *chbr = new chebyRate;

  cout << "kcheb - "; showRevision();
  if (argc != 2) { cout << "usage: kcheb basename(.ext)\n"; return 1; }
  inpfln = argv[1];
  basfln = getBasePart(inpfln); extfln = getExtPart(inpfln);
  if (extfln == "") { inpfln = basfln + ".inp"; }
  csvfln = basfln + "_kcheb_out.csv";

  // --- read in input
  inf.open(inpfln.c_str());
  if (!inf) { cout << "failed to open [" << inpfln << "].\n"; return 1; }
  while (getlParToksKey(inf, key, tkns, ntkns, slin) == NORM) {
    if (key == "tempCheb") {
      if (ntkns < 3) {
        cout << "Insufficient number of values for key [tempCheb].\n";
        return 1;
      }
      Tmin = atof(tkns[1].c_str()); Tmax = atof(tkns[2].c_str());
      if ((Tmin >= Tmax) || (Tmin <= 0.))
        { cout << "Invalid values for tempCheb.\n"; return 1; }
    } else if (key == "pressCheb") {
      if (ntkns < 3) {
        cout << "Insufficient number of values for key [pressCheb].\n";
        return 1;
      }
      Pmin = atof(tkns[1].c_str()); Pmax = atof(tkns[2].c_str());
      if ((Pmin >= Pmax) || (Pmin <= 0.))
        { cout << "Invalid values for pressCheb.\n"; return 1; }
    } else if (key.substr(0, 4) == "temp") {
      if (tlst.addFrom(slin) == ERRE) { return 1; }
    } else if (key.substr(0, 5) == "press") {
      if (plst.addFrom(slin) == ERRE) { return 1; }
    } else if (key == "coefTable{") {
      if (getlParToks(inf, tkns, ntkns, slin) == ERRE)
        { cout << "Invalid coefTable.\n"; return 1; }
      if (ntkns != 2) { cout << "Invalid nParT nParP.\n"; return 1; }
      nParT = atoi(tkns[0].c_str()); nParP = atoi(tkns[1].c_str());
      if (nParT <= 0) { cout << "nParT must be > 0.\n"; return 1; }
      if (nParP <= 0) { cout << "nParP must be > 0.\n"; return 1; }
      coef.clear();
      for (it = 0; it < nParT; it++) {
        if (getlParToks(inf, tkns, ntkns, slin) == ERRE)
          { cout << "Invalid coefTable.\n"; return 1; }
        if (ntkns != nParP)
          { cout << "Invalid row of coefTable.\n"; return 1; }
        crow.clear();
        for (ip = 0; ip < nParP; ip++)
          { crow.push_back(atof(tkns[ip].c_str())); }
        coef.push_back(crow);
      }
      if (getlParToks(inf, tkns, ntkns, slin) == ERRE)
        { cout << "Invalid coefTable.\n"; return 1; }
      if (tkns[0] != "}") { cout << "Invalid coefTable.\n"; return 1; }
    } else { cout << "Warning: Invalid key [" << key << "].\n"; }
  }
  inf.close();

  // --- check for input
  if (Tmin < 0.) { cout << "No [tempCheb] input found.\n"; return 1; }
  if (Pmin < 0.) { cout << "No [pressCheb] input found.\n"; return 1; }
  if (coef.empty()) { cout << "No [coefTable] input found.\n"; return 1; }
  if (tlst.empty()) { cout << "No temperature input found.\n"; return 1; }
  if (plst.empty()) { cout << "No pressure input found.\n"; return 1; }

  // --- do calculation
  reverse(tlst.begin(), tlst.end());
  reverse(plst.begin(), plst.end());
  if (chbr->init(nParT, nParP, Tmin, Tmax, Pmin, Pmax, coef) == ERRE)
    { return 1; }
  cof.open(csvfln.c_str());
  if (!cof)
    { cout << "failed to open [" << csvfln << "].\n"; return ERRE; }
  cof << "T[K],P[atm],k\n";
  for (iTemp = 0; iTemp < tlst.size(); iTemp++) {
    T = tlst[iTemp];
    for (iPress = 0; iPress < plst.size(); iPress++) {
      P = plst[iPress]; Patm = P / 760.;
      k = chbr->calc(T, Patm);
      cof << T << "," << Patm << "," << k << endl;
    }
  }
  cof.close();
  return 0;
}

