/*========================================================================
     "dislit" - steady-state dissociation linear problem solver
                 for multiple-well system (starting from single-well)
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2014 by A. Miyoshi, Univ. Tokyo
                                  created: Oct.  30, 2009
                                   edited: Nov.  22, 2009
                                   edited: Jan.  25, 2010
        - added kEthInt and kEthOut input
                                   edited: June   7, 2010
        - added alphaTdep input
                                   edited: Aug.  29, 2010
        - added HPL vector output
                                   edited: June   6, 2011
        - added truncate function
                              last edited: May   22, 2014
--------------------------------------------------------------------------
 usage: dislit basename(.ext)
--------------------------------------------------------------------------
 - If no extension is given, default extension '.inp' is assumed
 - output is written in [basename]_dislit_out.csv
 - HPL rate is written in [basename]_dislit_hpl.csv
 - when 'output vector' is set, steady-state vectors are written
    in [basename]_dislit_vec.csv
========================================================================*/

#include "ssulibc.h"

void outputHeader(ostream &os, listWells &lw);
void outputResults(ostream &os, listWells &lw);

/*===== main function ==================================================*/

int main(int argc, char *argv[]) {
  int inWell, ntkns, idxT, idxP, posw, outVec = false, nReactWell = 0;
  int err3set = false, outHPL = false, HPLonly = false, itkn, rDLS;
  int revPressOrd = false, alphaTset = false, inval_trunc = 0;
  double err3inp, a1000inp, aTEinp;
  string ctrlfln, basfln, extfln, csvfln, hplfln, hpvfln, vecfln, slin, key;
  ifstream inf;
  ofstream cof, hof, hvf, vcf;
  vector<string> tkns;
  tempList tlst;
  pressList plst;
  connex cnx;
  well molecw;
  listWells lstw;
  umolProb *ump = new umolProb;   // large object cannot be an autovariable

  cout << "dislit - "; showRevision();
  if (argc != 2) { cout << "usage: dislit basename(.ext)\n"; return 1; }
  ctrlfln = argv[1];
  basfln = getBasePart(ctrlfln); extfln = getExtPart(ctrlfln);
  if (extfln == "") { ctrlfln = basfln + ".inp"; }
  csvfln = basfln + "_dislit_out.csv";
  hplfln = basfln + "_dislit_hpl.csv";
  hpvfln = basfln + "_dislit_hplvec.csv";
  vecfln = basfln + "_dislit_vec.csv";
  ump->probtype = umolProb::LinSS;

  // --- read in control input
  inf.open(ctrlfln.c_str());
  if (!inf) { cout << "failed to open [" << ctrlfln << "].\n"; return 1; }
  inWell = false;
  while (getlParToksKey(inf, key, tkns, ntkns, slin) == NORM) {
    if (inWell) {
      if (key == "}") {
        if (lstw.add(molecw) == ERRE) {
          cout << "Duplicated well-index [" << molecw.index << "].\n";
          return 1;
        }
        inWell = false;
      } else if (key == "index") {
        if (ntkns < 2) { cout << "No value for key [index].\n"; return 1; }
        molecw.index = atoi(tkns[1].c_str());
      } else if (key == "filename") {
        if (ntkns < 2) { cout << "No value for key [filename].\n"; return 1; }
        molecw.fname = tkns[1];
      } else if (key == "offset") {
        if (ntkns < 2) { cout << "No value for key [offset].\n"; return 1; }
        molecw.offset = atoi(tkns[1].c_str());
      } else if (key == "connect") {
        if (ntkns < 3)
          { cout << "Two values needed for key [connect].\n"; return 1; }
        cnx.toWellIdx = atoi(tkns[1].c_str());
        cnx.chan = atoi(tkns[2].c_str()) - 1;
        molecw.connect.push_back(cnx);

/*     --- disabled 12-Nov-2009

      } else if (key == "thRhoGrad") {
        if (ntkns < 2) { cout << "No value for key [thRhoGrad].\n"; return 1; }
        molecw.thRhoGrad = atof(tkns[1].c_str());
      } else if (key == "redExpon") {
        if (ntkns < 2) { cout << "No value for key [redExpon].\n"; return 1; }
        molecw.redExpFix = atof(tkns[1].c_str());
*/
      } else if (key == "reactant") {
        molecw.isReactant = true;
      } else if (key == "truncate") {
        if (ntkns < 2) { cout << "No value for key [truncate].\n"; return 1; }
        molecw.lowest = atoi(tkns[1].c_str());
      } else { cout << "Warning: Invalid key [" << key << "].\n"; }
    } else {
      if (key == "well{") {
        molecw.clear();
        molecw.index = lstw.autoIndex();
        inWell = true;
      } else if (key.substr(0, 4) == "temp") {      // temperature list input
        if (tlst.addFrom(slin) == ERRE) { return 1; }
      } else if (key.substr(0, 5) == "press") {     // pressure list input
        if (plst.addFrom(slin) == ERRE) { return 1; }
      } else if (key == "revPressOrd") {
        revPressOrd = true;
      } else if (key == "maxCyc") {
        if (ntkns < 2) { cout << "No value for key [maxCyc].\n"; return 1; }
        ump->maxCycLinSs = atoi(tkns[1].c_str());
      } else if (key == "topCut") {
        if (ntkns < 2) { cout << "No value for key [topCut].\n"; return 1; }
        lstw.topCut = atoi(tkns[1].c_str());
      } else if (key == "err3") {
        if (ntkns < 2) { cout << "No value for key [err3].\n"; return 1; }
        err3inp = atof(tkns[1].c_str()); err3set = true;
      } else if (key == "alphaTdep") {
        if (ntkns < 3) { cout << "No value for key [alphaTdep].\n"; return 1; }
        a1000inp = atof(tkns[1].c_str()); aTEinp = atof(tkns[2].c_str());
        alphaTset = true;
      } else if (key == "kEthInt") {
        if (ntkns < 2) { cout << "No value for key [kEthInt].\n"; return 1; }
        lstw.kEthInt = atof(tkns[1].c_str());
      } else if (key == "kEthOut") {
        if (ntkns < 2) { cout << "No value for key [kEthOut].\n"; return 1; }
        lstw.kEthOut = atof(tkns[1].c_str());
      } else if (key == "verbose") {
        if (ntkns < 2) { cout << "No value for key [verbose].\n"; return 1; }
        ump->verbose = atoi(tkns[1].c_str());
      } else if (key == "output") {
        if (ntkns < 2) { cout << "Warning: No value for key [output].\n"; }
        else {
          for (itkn = 1; itkn < ntkns; itkn++) {
            if (tkns[itkn] == "vector") {
              outVec = true;
            } else if (tkns[itkn] == "HPL") {
              outHPL = true;
            } else if (tkns[itkn] == "HPLonly") {
              outHPL = true; HPLonly = true;
            } else {
              cout << "Warning: Invalid value [" << tkns[1]
                   << "] for key [output].\n";
            }
          }
        }
      } else { cout << "Warning: Invalid key [" << key << "].\n"; }
    }
  }
  inf.close();

  // --- check for reactant well
  for (posw = 0; posw < lstw.size(); posw++) {
    if (lstw[posw].isReactant) {
      nReactWell++; lstw.posReactWell = posw;
      if (lstw[posw].lowest > 0) { inval_trunc = 1; }
    }
  }
  if (nReactWell > 1) { cout << "Too may reactant-wells.\n"; return 1; }
  else if (nReactWell == 0) {
    for (posw = 0; posw < lstw.size(); posw++) {
      if (lstw[posw].lowest > 0) { inval_trunc = 2; }
    }
  }
  if (inval_trunc == 1)
    { cout << "Reactant-well cannot be truncated.\n"; return 1; }
  else if (inval_trunc == 2)
    { cout << "No truncation allowed for all-well-reactants problem.\n"; return 1; }

  // --- read in Unimol RRKMTH outputs
  if (lstw.readUnimol() == ERRE) { return 1; }
  if (err3set) { lstw.setErr3(err3inp); }
  if (alphaTset) { lstw.setAlpPar(a1000inp, aTEinp); }
  if (tlst.empty()) { lstw.mergedTlist(tlst); }
  reverse(tlst.begin(), tlst.end());
  if (plst.empty()) { lstw.mergedPlist(plst); }
  if (!revPressOrd) { reverse(plst.begin(), plst.end()); }
  lstw.printParETP(cout, ump->verbose);

  // --- do calculations
  if (ump->initMW(lstw) == ERRE) { return 1; }
   // --- HPL rate
  if (outHPL) {
    hof.open(hplfln.c_str());
    if (!hof) { cout << "failed to open [" << hplfln << "].\n"; return 1; }
    if (outVec) {
      hvf.open(hpvfln.c_str());
      if (!hvf) { cout << "failed to open [" << hpvfln << "].\n"; return 1; }
    }
    outputHeader(hof, lstw);
    ump->HPL = true;
    for (idxT = 0; idxT < tlst.size(); idxT++) {
      ump->calcHPLMW(lstw, tlst[idxT]);
      hof << ump->T << ",inf";
      outputResults(hof, lstw);
      if (outVec) { ump->outputVectorMW(hvf, lstw); }
    }
    ump->HPL = false;
    if (outVec) { hvf.close(); }
    hof.close();
  }
  if (HPLonly) { return 0; }
  cof.open(csvfln.c_str());
  if (!cof) { cout << "failed to open [" << csvfln << "].\n"; return 1; }
  if (outVec) {
    vcf.open(vecfln.c_str());
    if (!vcf) { cout << "failed to open [" << vecfln << "].\n"; return 1; }
  }
   // --- main loop
  outputHeader(cof, lstw);
  for (idxT = 0; idxT < tlst.size(); idxT++) {
    if (lstw.posReactWell < 0) { ump->setBoltzRflxMWall(lstw, tlst[idxT]); }
    else { ump->setBoltzRflxMWsel(lstw, tlst[idxT]); }
    for (idxP = 0; idxP < plst.size(); idxP++) {
      cout << "T = " << tlst[idxT] << " [K], p = " << plst[idxP] << " [Torr]:\n";
      rDLS = ump->driveLinSsMWposv(lstw, tlst[idxT], plst[idxP]);
      if (rDLS == ERRE) { return 1; }
      cof << ump->T << ',' << ump->p;
      if (rDLS == ITRERR) {
        cof << ",*** MAXCYCLE ***\n";
      } else {
        outputResults(cof, lstw);
        if (outVec) { ump->outputVectorMW(vcf, lstw); }
      }
    }
  }
  if (outVec) { vcf.close(); }
  cof.close();
  return 0;
}

/*----- output header of a csv file ------------------------------------*/

void outputHeader(ostream &os, listWells &lw) {
  int ch, posw, toposw;
  string s_outgo = "out-going-";
  if (lw.nTotStbW > 0) { s_outgo = "out-going-&-stab-"; }
   // - 1st line
  os << ",," << s_outgo << "rates";
  for (ch = 0; ch < lw.nTotOCh + lw.nTotStbW; ch++) { os << ','; }
  if (lw.nTotOCh + lw.nTotStbW > 0) { os << "," << s_outgo << "fractions"; }
  for (ch = 0; ch < lw.nTotOCh + lw.nTotStbW - 1; ch++) { os << ','; }
  if (lw.nTotCh - lw.nTotOCh > 0) { os << ",internal-rates"; }
  for (ch = 0; ch < lw.nTotCh - lw.nTotOCh - 1; ch++) { os << ','; }
  os << ",internal-populations";
  for (posw = 1; posw < lw.size(); posw++) { os << ','; }
  os << endl;
   // - 2nd line
  os << ',';
  for (posw = 0; posw < lw.size(); posw++) {
    if (lw[posw].nOutCh + lw[posw].nStab > 0) {
      os << ",well" << lw[posw].index;
      if (lw[posw].isReactant) { os << "(reactant)"; }
    }
    for (ch = 0; ch < lw[posw].nOutCh + lw[posw].nStab - 1; ch++)
      { os << ','; }
  }
  os << ",ktot";
  for (posw = 0; posw < lw.size(); posw++) {
    if (lw[posw].nOutCh + lw[posw].nStab > 0) {
      os << ",well" << lw[posw].index;
      if (lw[posw].isReactant) { os << "(reactant)"; }
    }
    for (ch = 0; ch < lw[posw].nOutCh + lw[posw].nStab - 1; ch++)
      { os << ','; }
  }
  for (posw = 0; posw < lw.size(); posw++) {
    if (lw[posw].nchan - lw[posw].nOutCh > 0) {
      os << ",well" << lw[posw].index;
      if (lw[posw].isReactant) { os << "(reactant)"; }
    }
    for (ch = 0; ch < lw[posw].nchan - lw[posw].nOutCh - 1; ch++)
      { os << ','; }
  }
  for (posw = 0; posw < lw.size(); posw++) {
    os << ",well" << lw[posw].index;
    if (lw[posw].isReactant) { os << "(reactant)"; }
  }
  os << endl;
   // - 3rd line
  os << "T[K],P[Torr]";
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1) { os << ",k" << (ch + 1); }
    }
    if (lw[posw].nStab > 0) { os << ",kstab"; }
  }
  os << ',';
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1) { os << ",f" << (ch + 1); }
    }
    if (lw[posw].nStab > 0) { os << ",fstab"; }
  }
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if ((toposw = lw[posw].chInf[ch]) >= 0)
        { os << ",k" << (ch + 1) << "(to" << lw[toposw].index << ')'; }
    }
  }
  for (posw = 0; posw < lw.size(); posw++) { os << ','; }
  os << endl;
}

/*----- output results to a csv file -----------------------------------*/

void outputResults(ostream &os, listWells &lw) {
  int posw, ch;
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1)
        { os << ',' << lw[posw].current.k[ch]; }
    }
    if (lw[posw].nStab > 0) { os << ',' << lw[posw].current.kstab; }
  }
  os << ',' << lw.koutTot;
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1)
        { os << ',' << lw[posw].current.k[ch] / lw.koutTot; }
    }
    if (lw[posw].nStab > 0) { os << ',' << lw[posw].current.kstab / lw.koutTot; }
  }
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] >= 0)
        { os << ',' << lw[posw].current.k[ch]; }
    }
  }
  for (posw = 0; posw < lw.size(); posw++)
    { os << ',' << lw[posw].current.fpop; }
  os << endl;
}

