/*========================================================================
     "catime" - time-dependent chemical-activation problem solver
                 for multiple-well system
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2011 by A. Miyoshi, Univ. Tokyo
                                  created: May   18, 2011
                              last edited: May   31, 2011
--------------------------------------------------------------------------
 usage: catime basename(.ext)
--------------------------------------------------------------------------
 - If no extension is given, default extension '.inp' is assumed
 - output is written in [basename]_catime_vec.csv
========================================================================*/

#include "ssulibc.h"

/*===== main function ==================================================*/

int main(int argc, char *argv[]) {
  int inWell, ntkns, idxT, idxP, err3set = false, itkn, alphaTset = false;
  double err3inp, a1000inp, aTEinp;
  string ctrlfln, basfln, extfln, vecfln, slin, key;
  ifstream inf;
  ofstream vcf;
  vector<string> tkns;
  tempList tlst;
  pressList plst;
  timeList timl;
  connex cnx;
  well molecw;
  listWells lstw;
  umolProb *ump = new umolProb;   // large object cannot be an autovariable

  cout << "catime - "; showRevision();
  if (argc != 2) { cout << "usage: catime basename(.ext)\n"; return 1; }
  ctrlfln = argv[1];
  basfln = getBasePart(ctrlfln); extfln = getExtPart(ctrlfln);
  if (extfln == "") { ctrlfln = basfln + ".inp"; }
  vecfln = basfln + "_catime_vec.csv";
  ump->probtype = umolProb::TimEv;

  // --- read in control input
  inf.open(ctrlfln.c_str());
  if (!inf) { cout << "failed to open [" << ctrlfln << "].\n"; return 1; }
  inWell = false;
  while (getlParToksKey(inf, key, tkns, ntkns, slin) == NORM) {
    if (inWell) {
      if (key == "}") {
        if (lstw.add(molecw) == ERRE) {
          cout << "Duplicated well-index [" << molecw.index << "].\n";
          return 1;
        }
        inWell = false;
      } else if (key == "index") {
        if (ntkns < 2) { cout << "No value for key [index].\n"; return 1; }
        molecw.index = atoi(tkns[1].c_str());
      } else if (key == "filename") {
        if (ntkns < 2) { cout << "No value for key [filename].\n"; return 1; }
        molecw.fname = tkns[1];
      } else if (key == "offset") {
        if (ntkns < 2) { cout << "No value for key [offset].\n"; return 1; }
        molecw.offset = atoi(tkns[1].c_str());
      } else if (key == "connect") {
        if (ntkns < 3)
          { cout << "Two values needed for key [connect].\n"; return 1; }
        cnx.toWellIdx = atoi(tkns[1].c_str());
        cnx.chan = atoi(tkns[2].c_str()) - 1;
        molecw.connect.push_back(cnx);
      } else if (key == "recombChan") {
        if (ntkns < 2) { cout << "No value for key [recombChan].\n"; return 1; }
        molecw.caWell = true;
        molecw.actchan = atoi(tkns[1].c_str()) - 1;
      } else if (key == "truncate") {
        if (ntkns < 2) { cout << "No value for key [truncate].\n"; return 1; }
        molecw.lowest = atoi(tkns[1].c_str());
      } else { cout << "Warning: Invalid key [" << key << "].\n"; }
    } else {
      if (key == "well{") {
        molecw.clear();
        molecw.index = lstw.autoIndex();
        inWell = true;
      } else if (key.substr(0, 4) == "temp") {      // temperature list input
        if (tlst.addFrom(slin) == ERRE) { return 1; }
      } else if (key.substr(0, 5) == "press") {     // pressure list input
        if (plst.addFrom(slin) == ERRE) { return 1; }
      } else if (key.substr(0, 4) == "time") {      // time list input
        if (timl.addFrom(slin) == ERRE) { return 1; }
      } else if (key == "topCut") {
        if (ntkns < 2) { cout << "No value for key [topCut].\n"; return 1; }
        lstw.topCut = atoi(tkns[1].c_str());
      } else if (key == "err3") {
        if (ntkns < 2) { cout << "No value for key [err3].\n"; return 1; }
        err3inp = atof(tkns[1].c_str()); err3set = true;
      } else if (key == "alphaTdep") {
        if (ntkns < 3) { cout << "No value for key [alphaTdep].\n"; return 1; }
        a1000inp = atof(tkns[1].c_str()); aTEinp = atof(tkns[2].c_str());
        alphaTset = true;
      } else if (key == "kEthInt") {
        if (ntkns < 2) { cout << "No value for key [kEthInt].\n"; return 1; }
        lstw.kEthInt = atof(tkns[1].c_str());
      } else if (key == "kEthOut") {
        if (ntkns < 2) { cout << "No value for key [kEthOut].\n"; return 1; }
        lstw.kEthOut = atof(tkns[1].c_str());
      } else if (key == "verbose") {
        if (ntkns < 2) { cout << "No value for key [verbose].\n"; return 1; }
        ump->verbose = atoi(tkns[1].c_str());
      } else if (key == "output") {
        if (ntkns < 2) { cout << "Warning: No value for key [output].\n"; }
        else {
          for (itkn = 1; itkn < ntkns; itkn++) {
            if (tkns[itkn] == "vector") {
              // - vector output is default
            } else {
              cout << "Warning: Invalid value [" << tkns[1]
                   << "] for key [output].\n";
            }
          }
        }
      } else if (key == "condition") {
        if (ntkns < 2) { cout << "Warning: No value for key [condition].\n"; }
        else {
          if (tkns[1] == "const") { ump->timeevcond = umolProb::TEconst; }
          else if (tkns[1] == "init") { ump->timeevcond = umolProb::TEinit; }
          else {
            cout << "Warning: Invalid value [" << tkns[1]
                 << "] for key [condition].\n";
          }
        }
      } else if (key == "sizeLSDSRW") {
        if (ntkns < 2) { cout << "No value for key [sizeLSDSRW].\n"; return 1; }
        ump->sizeLSDSRW = atof(tkns[1].c_str());
      } else { cout << "Warning: Invalid key [" << key << "].\n"; }
    }
  }
  inf.close();

  // --- read in Unimol RRKMTH outputs
  if (lstw.readUnimol() == ERRE) { return 1; }
  if (lstw.checkCA() == ERRE) { return 1; }
  if (err3set) { lstw.setErr3(err3inp); }
  if (alphaTset) { lstw.setAlpPar(a1000inp, aTEinp); }
  if (tlst.empty()) { lstw.mergedTlist(tlst); }
  reverse(tlst.begin(), tlst.end());
  if (plst.empty()) { lstw.mergedPlist(plst); }
  reverse(plst.begin(), plst.end());
  if (timl.empty()) { timl.setDefault(); }
  timl.addZero();
  lstw.printParETP(cout, ump->verbose);
  lstw.printParCA(cout);

  // --- do calculations
  if (ump->initMW(lstw) == ERRE) { return 1; }
  vcf.open(vecfln.c_str());
  if (!vcf) { cout << "failed to open [" << vecfln << "].\n"; return 1; }
   // --- main loop
  for (idxT = 0; idxT < tlst.size(); idxT++) {
    for (idxP = 0; idxP < plst.size(); idxP++) {
      cout << "T = " << tlst[idxT] << " [K], p = " << plst[idxP] << " [Torr]:\n";
      if (ump->setMatMW(lstw, tlst[idxT], plst[idxP]) == ERRE) { return 1; }
      if (ump->solveTimeEvMW(lstw, timl) == ERRE) { return 1; }
      ump->outputVecTimEvMW(vcf, lstw, timl);
    }
  }
  vcf.close();
  return 0;
}

