/*========================================================================
     "carate" - steady-state chemical-activation problem solver
                 for multiple-well system
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2010 by A. Miyoshi, Univ. Tokyo
                                  created: Oct.   1, 2009
                                   edited: Nov.  14, 2009
                                   edited: Jan.  25, 2010
        - added kEthInt and kEthOut input
                                   edited: June   7, 2010
        - added alphaTdep input
                                   edited: Aug.  29, 2010
        - added Rflx vector output
                              last edited: June   6, 2011
--------------------------------------------------------------------------
 usage: carate basename(.ext)
--------------------------------------------------------------------------
 - If no extension is given, default extension '.inp' is assumed
 - output is written in [basename]_carate_out.csv
========================================================================*/

#include "ssulibc.h"

void outputHeader(ostream &os, listWells &lw);
void outputResults(ostream &os, listWells &lw);

/*===== main function ==================================================*/

int main(int argc, char *argv[]) {
  int inWell, ntkns, idxT, idxP, outVec = false, err3set = false, itkn;
  int alphaTset = false;
  double err3inp, a1000inp, aTEinp;
  string ctrlfln, basfln, extfln, csvfln, rfvfln, vecfln, slin, key;
  ifstream inf;
  ofstream cof, rvf, vcf;
  vector<string> tkns;
  tempList tlst;
  pressList plst;
  connex cnx;
  well molecw;
  listWells lstw;
  umolProb *ump = new umolProb;   // large object cannot be an autovariable

  cout << "carate - "; showRevision();
  if (argc != 2) { cout << "usage: carate basename(.ext)\n"; return 1; }
  ctrlfln = argv[1];
  basfln = getBasePart(ctrlfln); extfln = getExtPart(ctrlfln);
  if (extfln == "") { ctrlfln = basfln + ".inp"; }
  csvfln = basfln + "_carate_out.csv";
  rfvfln = basfln + "_carate_rfxvec.csv";
  vecfln = basfln + "_carate_vec.csv";
  ump->probtype = umolProb::LinEq;

  // --- read in control input
  inf.open(ctrlfln.c_str());
  if (!inf) { cout << "failed to open [" << ctrlfln << "].\n"; return 1; }
  inWell = false;
  while (getlParToksKey(inf, key, tkns, ntkns, slin) == NORM) {
    if (inWell) {
      if (key == "}") {
        if (lstw.add(molecw) == ERRE) {
          cout << "Duplicated well-index [" << molecw.index << "].\n";
          return 1;
        }
        inWell = false;
      } else if (key == "index") {
        if (ntkns < 2) { cout << "No value for key [index].\n"; return 1; }
        molecw.index = atoi(tkns[1].c_str());
      } else if (key == "filename") {
        if (ntkns < 2) { cout << "No value for key [filename].\n"; return 1; }
        molecw.fname = tkns[1];
      } else if (key == "offset") {
        if (ntkns < 2) { cout << "No value for key [offset].\n"; return 1; }
        molecw.offset = atoi(tkns[1].c_str());
      } else if (key == "connect") {
        if (ntkns < 3)
          { cout << "Two values needed for key [connect].\n"; return 1; }
        cnx.toWellIdx = atoi(tkns[1].c_str());
        cnx.chan = atoi(tkns[2].c_str()) - 1;
        molecw.connect.push_back(cnx);
      } else if (key == "recombChan") {
        if (ntkns < 2) { cout << "No value for key [recombChan].\n"; return 1; }
        molecw.caWell = true;
        molecw.actchan = atoi(tkns[1].c_str()) - 1;
      } else if (key == "truncate") {
        if (ntkns < 2) { cout << "No value for key [truncate].\n"; return 1; }
        molecw.lowest = atoi(tkns[1].c_str());

/*     --- disabled 12-Nov-2009

      } else if (key == "thRhoGrad") {
        if (ntkns < 2) { cout << "No value for key [thRhoGrad].\n"; return 1; }
        molecw.thRhoGrad = atof(tkns[1].c_str());
      } else if (key == "redExpon") {
        if (ntkns < 2) { cout << "No value for key [redExpon].\n"; return 1; }
        molecw.redExpFix = atof(tkns[1].c_str());
*/
      } else { cout << "Warning: Invalid key [" << key << "].\n"; }
    } else {
      if (key == "well{") {
        molecw.clear();
        molecw.index = lstw.autoIndex();
        inWell = true;
      } else if (key.substr(0, 4) == "temp") {      // temperature list input
        if (tlst.addFrom(slin) == ERRE) { return 1; }
      } else if (key.substr(0, 5) == "press") {     // pressure list input
        if (plst.addFrom(slin) == ERRE) { return 1; }
      } else if (key == "topCut") {
        if (ntkns < 2) { cout << "No value for key [topCut].\n"; return 1; }
        lstw.topCut = atoi(tkns[1].c_str());
      } else if (key == "err3") {
        if (ntkns < 2) { cout << "No value for key [err3].\n"; return 1; }
        err3inp = atof(tkns[1].c_str()); err3set = true;
      } else if (key == "alphaTdep") {
        if (ntkns < 3) { cout << "No value for key [alphaTdep].\n"; return 1; }
        a1000inp = atof(tkns[1].c_str()); aTEinp = atof(tkns[2].c_str());
        alphaTset = true;
      } else if (key == "kEthInt") {
        if (ntkns < 2) { cout << "No value for key [kEthInt].\n"; return 1; }
        lstw.kEthInt = atof(tkns[1].c_str());
      } else if (key == "kEthOut") {
        if (ntkns < 2) { cout << "No value for key [kEthOut].\n"; return 1; }
        lstw.kEthOut = atof(tkns[1].c_str());
      } else if (key == "verbose") {
        if (ntkns < 2) { cout << "No value for key [verbose].\n"; return 1; }
        ump->verbose = atoi(tkns[1].c_str());
      } else if (key == "output") {
        if (ntkns < 2) { cout << "Warning: No value for key [output].\n"; }
        else {
          for (itkn = 1; itkn < ntkns; itkn++) {
            if (tkns[itkn] == "vector") {
              outVec = true;
            } else {
              cout << "Warning: Invalid value [" << tkns[1]
                   << "] for key [output].\n";
            }
          }
        }
      } else { cout << "Warning: Invalid key [" << key << "].\n"; }
    }
  }
  inf.close();

  // --- read in Unimol RRKMTH outputs
  if (lstw.readUnimol() == ERRE) { return 1; }
  if (lstw.checkCA() == ERRE) { return 1; }
  if (err3set) { lstw.setErr3(err3inp); }
  if (alphaTset) { lstw.setAlpPar(a1000inp, aTEinp); }
  if (tlst.empty()) { lstw.mergedTlist(tlst); }
  reverse(tlst.begin(), tlst.end());
  if (plst.empty()) { lstw.mergedPlist(plst); }
  reverse(plst.begin(), plst.end());
  lstw.printParETP(cout, ump->verbose);
  lstw.printParCA(cout);

  // --- do calculations
  if (ump->initMW(lstw) == ERRE) { return 1; }
  cof.open(csvfln.c_str());
  if (!cof) { cout << "failed to open [" << csvfln << "].\n"; return 1; }
  if (outVec) {
    vcf.open(vecfln.c_str());
    if (!vcf) { cout << "failed to open [" << vecfln << "].\n"; return 1; }
    rvf.open(rfvfln.c_str());
    if (!rvf) { cout << "failed to open [" << rfvfln << "].\n"; return 1; }
  }
  outputHeader(cof, lstw);
   // --- main loop
  for (idxT = 0; idxT < tlst.size(); idxT++) {
    for (idxP = 0; idxP < plst.size(); idxP++) {
      cout << "T = " << tlst[idxT] << " [K], p = " << plst[idxP] << " [Torr]:\n";
      if (ump->setMatMWinvSgn(lstw, tlst[idxT], plst[idxP]) == ERRE) { return 1; }
      if (outVec) {
        if (idxP == 0) {
          ump->RFlxOut = true;
          ump->outputVectorMW(rvf, lstw);
          ump->RFlxOut = false;
        }
      }
      if (ump->solveLinEqMWposv(lstw) == ERRE) { return 1; }
      cof << ump->T << ',' << ump->p;
      outputResults(cof, lstw);
      if (outVec) { ump->outputVectorMW(vcf, lstw); }
    }
  }
  if (outVec) { rvf.close(); vcf.close(); }
  cof.close();
  return 0;
}

/*----- output header of a csv file ------------------------------------*/

void outputHeader(ostream &os, listWells &lw) {
  int ch, posw, toposw;
   // - 1st line
  os << ",,out-going-&-stab-fractions";
  for (ch = 0; ch < lw.nTotOCh + lw.size() - 1; ch++) { os << ','; }
  os << ",out-going-&-stab-rates";
  for (ch = 0; ch < lw.nTotOCh + lw.size(); ch++) { os << ','; }
  if (lw.nTotCh - lw.nTotOCh > 0) { os << ",internal-rates"; }
  for (ch = 0; ch < lw.nTotCh - lw.nTotOCh - 1; ch++) { os << ','; }
  os << ",internal-populations";
  for (posw = 1; posw < lw.size(); posw++) { os << ','; }
  os << endl;
   // - 2nd line
  os << ',';
  for (posw = 0; posw < lw.size(); posw++) {
    os << ",well" << lw[posw].index;
    for (ch = 0; ch < lw[posw].nOutCh; ch++) { os << ','; }
  }
  for (posw = 0; posw < lw.size(); posw++) {
    os << ",well" << lw[posw].index;
    for (ch = 0; ch < lw[posw].nOutCh; ch++) { os << ','; }
  }
  os << ",ktot";
  for (posw = 0; posw < lw.size(); posw++) {
    if (lw[posw].nchan - lw[posw].nOutCh > 0)
      { os << ",well" << lw[posw].index; }
    for (ch = 0; ch < lw[posw].nchan - lw[posw].nOutCh - 1; ch++)
      { os << ','; }
  }
  for (posw = 0; posw < lw.size(); posw++)
    { os << ",well" << lw[posw].index; }
  os << endl;
   // - 3rd line
  os << "T[K],P[Torr]";
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1) { os << ",f" << (ch + 1); }
      if ((lw[posw].caWell) && (ch == lw[posw].actchan)) { os << "(back)"; }
    }
    os << ",fstab";
  }
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1) { os << ",k" << (ch + 1); }
      if ((lw[posw].caWell) && (ch == lw[posw].actchan)) { os << "(back)"; }
    }
    os << ",kstab";
  }
  os << ',';
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if ((toposw = lw[posw].chInf[ch]) >= 0)
        { os << ",k" << (ch + 1) << "(to" << lw[toposw].index << ')'; }
    }
  }
  for (posw = 0; posw < lw.size(); posw++) { os << ','; }
  os << endl;
}

/*----- output results to a csv file -----------------------------------*/

void outputResults(ostream &os, listWells &lw) {
  int posw, ch;
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1)
        { os << ',' << lw[posw].current.k[ch] / lw.koutTot; }
    }
    os << ',' << lw[posw].current.kstab / lw.koutTot;
  }
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1)
        { os << ',' << lw[posw].current.k[ch]; }
    }
    os << ',' << lw[posw].current.kstab;
  }
  os << ',' << lw.koutTot;
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] >= 0)
        { os << ',' << lw[posw].current.k[ch]; }
    }
  }
  for (posw = 0; posw < lw.size(); posw++)
    { os << ',' << lw[posw].current.fpop; }
  os << endl;
}

