/*========================================================================
     "ssulibc" - common library classes
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2009 by A. Miyoshi, Univ. Tokyo
========================================================================*/

#ifndef _SSULIBC_H_
#define _SSULIBC_H_

#include "ssulibb.h"

/*========================================================================
     Common Library Classes
========================================================================*/

/*------------------------------------------------------------------------
     List of Temperatures
------------------------------------------------------------------------*/

class tempList : public vector<double> {
 protected:                                    // --- data members ---
  enum errcode { INVkey, INVstep, INVrange, INVparN, INVtemp };
 public:                                   // --- member functions ---
  tempList();
  void setDefault();
  int addFrom(string slin);
  void sortElimDup();
  int addRange(double st, double et, double dt);
  int addRangeVS(vector<string> &vs);
  int addRecipRange(double numr, double stt, double lst, double stp);
  int addRecipRangeVS(vector<string> &vs);
  int addGauChebGrd(double Tmin, double Tmax, int nT);
  int addGauChebGrdVS(vector<string> &vs);
  int addList(vector<double> &vd);
  int addListVS(vector<string> &vs);
  ostream &print(ostream &os);
  double maxTemp();
 protected:
  int errms(errcode erc, string sv = "");
};                                                  // --- globals ---
ostream &operator<<(ostream &os, tempList &tl);

/*------------------------------------------------------------------------
     List of Pressures
------------------------------------------------------------------------*/

// internal values are in Torr unit

#define ATMTOR  760.               // atm -> Torr
#define PATOR   0.007500616827     // Pa  -> Torr
#define KPATOR  7.500616827        // kPa -> Torr
#define MPATOR  7500.616827        // MPa -> Torr
#define BARTOR  750.0616827        // bar -> Torr

class pressList : public vector<double> {
 public:                                       // --- data members ---
  string unit;
  double convFact;
 protected:
  enum errcode { INVkey, INVstep, INVrange, INVparN, INVpress, INVunit };
 public:                                   // --- member functions ---
  pressList();
  void setDefault();
  int setUnit(string ui);
  int addFrom(string slin);
  void sortElimDup();
  int addRange(double sp, double ep, double dp);
  int addRangeVS(vector<string> &vs);
  int addLog10Range(double stt, double lst, double stp);
  int addLog10RangeVS(vector<string> &vs);
  int addGauChebGrd(double Pmin, double Pmax, int nP);
  int addGauChebGrdVS(vector<string> &vs);
  int addList(vector<double> &vd);
  int addListVS(vector<string> &vs);
  ostream &print(ostream &os);
  double maxPress();
 protected:
  int errms(errcode erc, string sv = "");
};                                                  // --- globals ---
ostream &operator<<(ostream &os, pressList &pl);

/*------------------------------------------------------------------------
     Well with Multiple Reaction Channels
      - corresponds to one isomer and one "rrkmth" output
------------------------------------------------------------------------*/

struct connex {
  int toWellIdx, chan;
};

class well {
 public:                                       // --- data members ---
  int index;
  int posBmat, caWell;
  int isReactant;               // reactant flag for multiwell eigen-prob
  vector<connex> connect;
  string fname;
  string title;
  int offset, lowest, upbound;  // offset, lowest grain, & upper-boundary
  int nchan, grsize;            // number of channels and grain size [cm-1]
  int ngrain, nreacg, nnonrg;   // numbers of grains, non-reactive grains
                                //  & reactive grains
  int actchan;                  // activation channel for chem-activation
  pressList press;
  tempList temp;
  double sigma, eps, redmass;   // in [angstrom], [K], and [amu]
  double alpha1000, alpTex;     // alpha at 1000 K & T-exponent
  double err3;                  // ERR3 to determine band width
  const static double DEF_thRhoGrad;
  const static double DEF_redExpFix;
  const static double MIN_redExp, MAX_redExp, STP_redExp;
  double thRhoGrad;             // threshold rho gradient
  double redExpFix;             // fixed reduction exponent
  vector<double> rho;
  vector<vector<double> > rate;
  struct {
    double T, p;                // current temp. [K], press. [Torr],
    double alpha, collfreq;     //  alpha [cm-1], and coll. freq. [s-1]
    double redExpon;            // reduction exponent
    double keigen, ktot, kstab, fstab;
    vector<double> k, f;
    double fpop;                // fraction of population (MW problem)
  } current;
  vector<int> chInf;
  int nOutCh;
 public:                                   // --- member functions ---
  well();
  void clear();
  int readFromUnimol();
  void printParETP(ostream &os, int vb);  // print ETP parameters
  void printParCA(ostream &os);           // ptint chem-activation pars
  int setCurrentTp(double Tin, double pin);
};

/*------------------------------------------------------------------------
     List of Wells
------------------------------------------------------------------------*/

class listWells : public vector<well> {
 public:                                       // --- data members ---
  int totSize, absTop, nTotCh, nTotOCh;
  int posReactWell;             // position of the reactant well
  int topCut;                   // number of top grains to be cut
  double koutTot, keigen;
  map<int, int> tableIndices;
 public:                                   // --- member functions ---
  listWells();
  void clear();
  int isValidIndex(int idx);
  int isUsedIndex(int idx);
  int autoIndex();                  // automatically generate an unused index
  int getPos(int idx);
  int add(well &w);
  int readUnimol();                 // read UNIMOL files, check, & align
  void mergedTlist(tempList &tl);         // get merged T-list
  void mergedPlist(pressList &pl);        // get merged p-list
  void printParETP(ostream &os, int vb);  // print ETP parameters
  void printParCA(ostream &os);           // ptint chem-activation pars
  int reduceSize(int maxs);               // reduce the total size
  void setPosBmat();                      // set posBmat in each well
  int checkCA();                          // check for chemical activation
  void setErr3(double e);                 // set err3
 protected:
  int checkConsistency();       // check consistency of wells
  int checkConnect();           // check for well-connections
  void alignSize();             // align the offset & upbound
  int checkEqkE();              // check for equilib. conditions of k(E)
  void setChanInfo();           // set chInf in wells
};

/*------------------------------------------------------------------------
     Unimolecular Problem Solver
------------------------------------------------------------------------*/

class umolProb {
 public:                                       // --- data members ---
  enum { MXSIZ =   7000,        // maximum size of the problem
         MXCOL =   2100,        // 30% of MXSIZ
         MXSZ2 =  14000,        // twice MXSIZ
         WMSIZ = 455000,        // DSYEVR work size = 65 * MXSIZ
         WISIZ =  70000 };      // MASIZ * 10
//
//  enum { MXSIZ =   5000,        // maximum size of the problem
//         MXCOL =   1500,        // 30% of MXSIZ
//         MXSZ2 =  10000,        // twice MXSIZ
//         WMSIZ = 325000,        // DSYEVR work size = 65 * MXSIZ
//         WISIZ =  50000 };      // MASIZ * 10
//
//  enum { MXSIZ =  10000,        // maximum size of the problem
//         MXCOL =   3000,        // 30% of MXSIZ
//         MXSZ2 =  20000,        // twice MXSIZ
//         WMSIZ = 650000,        // DSYEVR work size = 65 * MXSIZ
//         WISIZ = 100000 };      // MASIZ * 10
//
//  Total memory required for b,z,ab,v,r,s,d,work,ipiv,isuppz,iwork
//   = MXSIZ * ( MXSIZ * 1.3 + 75.5) * 8 / 1024 / 1024 (mb)
//  MXSIZ    memory
//   5000    251 MB
//   7000    490 MB
//  10000    998 MB
//
  enum { Eigen = 0, LinEq = 1, LinSS = 2, ItrSS = 3 } probtype;
  int maxCycLinSs, maxCycItrSs;
  int verbose;
  int psiz;                     // problem size
  int esiz;                     // number of calculated eigenvalues
  int noffdiag;                 // number of off-diagonal elements
  int posLNE;                   // pos. of the largest neg. eigenvalue
  double T, p;                  // temperature [K], pressure [Torr]
  double b[MXSIZ][MXSIZ];
  double z[MXCOL][MXSIZ];
//  double ab[MXSIZ][BDSIZ];
  double v[MXSIZ];
  double r[1][MXSIZ];           // right hand side for DPOSV/DPBSV
  double s[MXSIZ];              // similarity transformation vector S-1
  double d[MXSIZ];              // sum of the down-trans rate below lowest
  double work[WMSIZ];
  int ipiv[MXSIZ];
  int isuppz[MXSZ2];
  int iwork[WISIZ];
 public:                                   // --- member functions ---
  umolProb();
  void clear();
  void init1W(well &w);
                                   // set matrix for multiple well problem
  int initMW(listWells &lw);       // initialization for multi-well
                                   // set Boltzmann flux
  void calcHPL1W(well &w, double Tin);
  void calcHPLMW(listWells &lw, double Tin);
  void setBoltzRflxMWall(listWells &lw, double Tin);
  void setBoltzRflxMWsel(listWells &lw, double Tin);
  void setBoltzRflx1W(well &w, double Tin, int pos0);
                                   // LinSs & ItrSs problem drivers
  int driveLinSs1W(well &w, double Tin, double pin);
  int driveItrSs1W(well &w, double Tin, double pin);
  int driveLinSsMW(listWells &lw, double Tin, double pin);
  int driveItrSsMW(listWells &lw, double Tin, double pin);
                                   // setup matrix
  int setMat1W(well &w, double Tin, double pin);
  int setMatMW(listWells &lw, double Tin, double pin);
  int solveEigen1W(well &w);       // solve eigenvalue problem (single well)
//  int solveEigenBand1W(well &w);   //  (for banded matrix)
  int solveEigen1Wdsyevr(well &w);
  int solveLinEq1W(well &w);       // solve linear equation (single well)
  void outputVector1W(ostream &os, well &w);
  int solveEigenMW(listWells &lw); // solve eigenvalue problem (multi-well)
  int solveEigenMWdsyevr(listWells &lw);
  int solveLinEqMW(listWells &lw); // solve linear equation (multi-well)
  void outputVectorMW(ostream &os, listWells &lw);
  void printMat(ostream &os, int start, double fact);
  void outEigVal(ostream &os, int n);   // output largest n eigenvalues in v
 protected:
  int drvSetETP(well &w, int pos0, int &noff);          // setETPorb driver
  int setETProb(well &w, int pos0, int &noff);          // set P(E,E') in b
  void setRateSmat(well &w, int pos0);  // set k(E) in b & S-1 in s[]
  void setRate(well &w, int pos0);
  void setSmat(well &w, int pos0);
  void setCrossRate(listWells &lw);     // set non-diagonal k(E)
  void clearRflx();                     // clear r[][]
  void setRflxChan(well &w, int pos0);  // set r(E) in r from channel info.
  void restoreRflx(well &w, int pos0);  // restore r from v[]
  void storeRflx(well &w, int pos0, double stp);        // store r to v[]
  void storeRflxAll(double stp);        // store r to v[]
  void restoreSSG();                    // restore r from v[]
  void storeSSG();                      // store r to v[]
  int symmetMat();                      // make b-mat symmetric
  void applyStoR();                     // apply S to r
  void applyStoV();                     // apply S to v
  void makeMatBanded(well &w, int pos0);        // make b[][] banded
//  void prepABmat();                     // prepare ab for DSBEV/DPBSV
  void findLargeNegEigen();             // find the largest neg. eigenvalue
                                        // normalization of g (1-well)
  double normG1W(well &w, int pos0, double *g);
                                        // normalization of g (multi-well)
  double normGMW(listWells &lw, double *g);
                                        // integrate k(E)g (1-well)
  void integkEg1W(well &w, int pos0, double *g, double gnorm);
                                        // integrate k(E)g (multi-well)
  void integkEgMW(listWells &lw, double *g, double tgnorm);
  double calcDGdt();
  void addDGstep(double nabs, double stp);
};

#endif

