/*========================================================================
     "ssulibc" - common library classes
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2009 by A. Miyoshi, Univ. Tokyo
                                  created: Sept. 22, 2009 (from gpoplibc)
                              last edited: Dec.   8, 2009
========================================================================*/

#include "ssulibc.h"
#include "lapack_dsyev/wcpp_dsyev.h"
#include "lapack_dsyevr/wcpp_dsyevr.h"
// #include "lapack_dsbev/wcpp_dsbev.h"
#include "lapack_dsysv/wcpp_dsysv.h"

/*========================================================================
     Common Library Classes
========================================================================*/

/*------------------------------------------------------------------------
     List of Temperatures
------------------------------------------------------------------------*/

tempList::tempList() { clear(); }                 // --- (constructor) ---

void tempList::setDefault()                          // --- setDefault ---
  { clear(); addRange(300., 1500., 100.); }

int tempList::addFrom(string slin) {   // --- add from an input string ---
  string key;
  vector<string> tkns;
  tkns = splitTokens(slin, " ,\t");
  if (tkns.empty()) { return errms(INVparN); }
  key = tkns[0];
  if (key == "tempRange") { return addRangeVS(tkns); }
  else if (key == "tempRecipRange") { return addRecipRangeVS(tkns); }
  else if (key == "tempGauChebGrd") { return addGauChebGrdVS(tkns); }
  else if (key == "tempList") { return addListVS(tkns); }
  return errms(INVkey, key);
}

                          // --- sort and eliminate duplicated element ---
void tempList::sortElimDup() {
  int id, nelm = size();
  double dlast;
  vector<double> vd = *this;
  if (nelm < 2) { return; }
  sort(vd.begin(), vd.end());
  clear(); push_back(vd[0]); dlast = vd[0];
  for (id = 1; id < nelm; id++) {
    if ((vd[id] - dlast) > 1.e-10) { push_back(vd[id]); dlast = vd[id]; }
  }
}

                                    // --- add a range of temperatures ---
int tempList::addRange(double st, double et, double dt) {
  double t;
  if (dt < 1.e-6) { return errms(INVstep); }
  if ((st < 1.e-6) || (et < st)) { return errms(INVrange); }
  if (((et - st) / dt) > 1.e4) { return errms(INVstep); }
  for (t = st; t < (et + dt * .1); t += dt) { push_back(t); }
  sortElimDup();
  return NORM;
}

                                       // --- add a range from strings ---
int tempList::addRangeVS(vector<string> &vs) {
  // - assumes that vs[0] is the key "tempRange"
  double st, et, dt;
  if (vs.size() < 4) { return errms(INVparN); }
  st = atof(vs[1].c_str()); et = atof(vs[2].c_str());
  dt = atof(vs[3].c_str());
  return addRange(st, et, dt);
}

                            // --- add a range from reciprocal numbers ---
int tempList::addRecipRange(double numr, double stt, double lst,
 double stp) {
  double rt;
  if (stp < 1.e-12) { return errms(INVstep); }
  if ((stt < 1.e-12) || (lst < stt)) { return errms(INVrange); }
  if (((lst - stt) / stp) > 1.e4) { return errms(INVstep); }
  for (rt = stt; rt < (lst + stp * .1); rt += stp)
   { push_back(numr / rt); }
  sortElimDup();
  return NORM;
}

                 // --- add a range from reciprocal numbers in strings ---
int tempList::addRecipRangeVS(vector<string> &vs) {
  // - assumes that vs[0] is the key "tempRecipRange"
  double numr, stt, lst, stp;
  if (vs.size() < 5) { return errms(INVparN); }
  numr = atof(vs[1].c_str()); stt = atof(vs[2].c_str());
  lst = atof(vs[3].c_str()); stp = atof(vs[4].c_str());
  return addRecipRange(numr, stt, lst, stp);
}

                                // --- add Gauss-Chebyshev grid points ---
int tempList::addGauChebGrd(double Tmin, double Tmax, int nT) {
  int i;
  double Ttil, Tinv, Tmini = 1. / Tmin, Tmaxi = 1. / Tmax;
  for (i = 0; i < nT; i++) {
    Ttil = cos((2. * i + 1.) / 2. / nT * M_PI);
    Tinv = (Ttil * (Tmaxi - Tmini) + (Tmaxi + Tmini)) / 2.;
    push_back(1. / Tinv);
  }
  sortElimDup();
  return NORM;
}

                     // --- add Gauss-Chebyshev grid points in strings ---
int tempList::addGauChebGrdVS(vector<string> &vs) {
  int nT;
  double Tmin, Tmax;
  if (vs.size() < 4) { return errms(INVparN); }
  Tmin = atof(vs[1].c_str()); Tmax = atof(vs[2].c_str());
  nT = atoi(vs[3].c_str());
  return addGauChebGrd(Tmin, Tmax, nT);
}

                                     // --- add a list of temperatures ---
int tempList::addList(vector<double> &vd) {
  int id, nd = vd.size();
  for (id = 0; id < nd; id++) {
    if (vd[id] < 1.e-6) { return errms(INVtemp); }
    push_back(vd[id]);
  }
  sortElimDup();
  return NORM;
}

                          // --- add a list of temperatures in strings ---
int tempList::addListVS(vector<string> &vs) {
  // - assumes that vs[0] is the key "tempList"
  int is, ns = vs.size();
  vector<double> vd;
  vd.clear();
  for (is = 1; is < ns; is++) { vd.push_back(atof(vs[is].c_str())); }
  if (!vd.empty()) { return addList(vd); }
  return NORM;
}

ostream &tempList::print(ostream &os) {                   // --- print ---
  int itl, ntl = size();
  os << "tempList";
  for (itl = 0; itl < ntl; itl++) { os << " " << (*this)[itl]; }
  os << endl;
  return os;
}

double tempList::maxTemp()                              // --- maxTemp ---
  { return *max_element(begin(), end()); }

int tempList::errms(errcode erc, string sv) {    // --- error messages ---
  string msg = "tempList: ";
  switch(erc) {
    case INVkey: msg += "Invalid key[" + sv + "]."; break;
    case INVstep: msg += "Invalid (too small) step input."; break;
    case INVrange: msg += "Invalid range input."; break;
    case INVparN: msg += "Too small number of input parameters."; break;
    case INVtemp: msg += "Invalid temperature (< 0) input."; break;
  }
  cout << msg << endl;
  return ERRE;
}

ostream &operator<<(ostream &os, tempList &tl)    //------ operator << ---
  { return tl.print(os); }

/*------------------------------------------------------------------------
     List of Pressures
------------------------------------------------------------------------*/

pressList::pressList()                            // --- (constructor) ---
  { clear(); setUnit("Torr"); }

void pressList::setDefault() {                       // --- setDefault ---
  clear();
  setUnit("atm"); addLog10Range(-5., 5., 1.); setUnit("Torr");
}

int pressList::setUnit(string ui) {                     // --- setUnit ---
  string u = ui;
  strAllToLower(u);
  if (u == "torr") { unit = "Torr"; convFact = 1.; }
  else if (u == "atm") { unit = "atm"; convFact = ATMTOR; }
  else if (u == "pa") { unit = "Pa"; convFact = PATOR; }
  else if (u == "kpa") { unit = "kPa"; convFact = KPATOR; }
  else if (u == "mpa") { unit = "MPa"; convFact = MPATOR; }
  else if (u == "bar") { unit = "bar"; convFact = BARTOR; }
  else { return errms(INVunit, ui); }
  return NORM;
}

int pressList::addFrom(string slin) {  // --- add from an input string ---
  string key;
  vector<string> tkns;
  tkns = splitTokens(slin, " ,\t");
  if (tkns.empty()) { return errms(INVparN); }
  key = tkns[0];
  if (key == "pressRange") { return addRangeVS(tkns); }
  else if (key == "pressLog10Range") { return addLog10RangeVS(tkns); }
  else if (key == "pressGauChebGrd") { return addGauChebGrdVS(tkns); }
  else if (key == "pressList") { return addListVS(tkns); }
  else if (key == "pressUnit") { return setUnit(tkns[1]); }
  return errms(INVkey, key);
}

                          // --- sort and eliminate duplicated element ---
void pressList::sortElimDup() {
  int id, nelm = size();
  double dlast;
  vector<double> vd = *this;
  if (nelm < 2) { return; }
  sort(vd.begin(), vd.end());
  clear(); push_back(vd[0]); dlast = vd[0];
  for (id = 1; id < nelm; id++) {
    if ((vd[id] - dlast) > 1.e-64) { push_back(vd[id]); dlast = vd[id]; }
  }
}

                                       // --- add a range of pressures ---
int pressList::addRange(double sp, double ep, double dp) {
  double p;
  if (dp < 1.e-64) { return errms(INVstep); }
  if ((sp < 1.e-64) || (ep < sp)) { return errms(INVrange); }
  if (((ep - sp) / dp) > 1.e4) { return errms(INVstep); }
  for (p = sp; p < (ep + dp * .1); p += dp)
    { push_back(p * convFact); }
  sortElimDup();
  return NORM;
}

                                       // --- add a range from strings ---
int pressList::addRangeVS(vector<string> &vs) {
  // - assumes that vs[0] is the key "pressRange"
  double sp, ep, dp;
  if (vs.size() < 4) { return errms(INVparN); }
  sp = atof(vs[1].c_str()); ep = atof(vs[2].c_str());
  dp = atof(vs[3].c_str());
  return addRange(sp, ep, dp);
}

                                        // --- add a logarithmic range ---
int pressList::addLog10Range(double stt, double lst, double stp) {
  double lp;
  if (stp < 1.e-12) { return errms(INVstep); }
  if ((stt < -100.) || (lst < stt)) { return errms(INVrange); }
  if (((lst - stt) / stp) > 1.e4) { return errms(INVstep); }
  for (lp = stt; lp < (lst + stp * .1); lp += stp)
   { push_back(pow(10., lp) * convFact); }
  sortElimDup();
  return NORM;
}

                             // --- add a logarithmic range in strings ---
int pressList::addLog10RangeVS(vector<string> &vs) {
  // - assumes that vs[0] is the key "pressLog10Range"
  double stt, lst, stp;
  if (vs.size() < 4) { return errms(INVparN); }
  stt = atof(vs[1].c_str()); lst = atof(vs[2].c_str());
  stp = atof(vs[3].c_str());
  return addLog10Range(stt, lst, stp);
}

                                // --- add Gauss-Chebyshev grid points ---
int pressList::addGauChebGrd(double Pmin, double Pmax, int nP) {
  int i;
  double Ptil, logP, logPmin = log(Pmin), logPmax = log(Pmax);
  for (i = 0; i < nP; i++) {
    Ptil = cos((2. * i + 1.) / 2. / nP * M_PI);
    logP = (Ptil * (logPmax - logPmin) + (logPmax + logPmin)) / 2.;
    push_back(exp(logP) * convFact);
  }
  sortElimDup();
  return NORM;
}

                     // --- add Gauss-Chebyshev grid points in strings ---
int pressList::addGauChebGrdVS(vector<string> &vs) {
  int nP;
  double Pmin, Pmax;
  if (vs.size() < 4) { return errms(INVparN); }
  Pmin = atof(vs[1].c_str()); Pmax = atof(vs[2].c_str());
  nP = atoi(vs[3].c_str());
  return addGauChebGrd(Pmin, Pmax, nP);
}

                                        // --- add a list of pressures ---
int pressList::addList(vector<double> &vd) {
  int id, nd = vd.size();
  for (id = 0; id < nd; id++) {
    if (vd[id] < 1.e-64) { return errms(INVpress); }
    push_back(vd[id] * convFact);
  }
  sortElimDup();
  return NORM;
}

                             // --- add a list of pressures in strings ---
int pressList::addListVS(vector<string> &vs) {
  // - assumes that vs[0] is the key "pressList"
  int is, ns = vs.size();
  vector<double> vd;
  vd.clear();
  for (is = 1; is < ns; is++) { vd.push_back(atof(vs[is].c_str())); }
  if (!vd.empty()) { return addList(vd); }
  return NORM;
}

ostream &pressList::print(ostream &os) {                  // --- print ---
  int ipl, npl = size();
  os << "pressList";
  for (ipl = 0; ipl < npl; ipl++) { os << " " << (*this)[ipl]; }
  os << endl;
  return os;
}

double pressList::maxPress()                           // --- maxPress ---
  { return *max_element(begin(), end()); }

int pressList::errms(errcode erc, string sv) {   // --- error messages ---
  string msg = "pressList: ";
  switch(erc) {
    case INVkey: msg += "Invalid key [" + sv + "]."; break;
    case INVstep: msg += "Invalid (too small) step input."; break;
    case INVrange: msg += "Invalid range input."; break;
    case INVparN: msg += "Too small number of input parameters."; break;
    case INVpress: msg += "Invalid pressure (< 0) input."; break;
    case INVunit: msg += "Invalid unit [" + sv + "]."; break;
  }
  cout << msg << endl;
  return ERRE;
}

ostream &operator<<(ostream &os, pressList &pl)   //------ operator << ---
  { return pl.print(os); }

/*------------------------------------------------------------------------
     Well with Multiple Reaction Channels
      - corresponds to one isomer and one "rrkmth" output
------------------------------------------------------------------------*/

const double well::DEF_thRhoGrad = 5.;
const double well::DEF_redExpFix = -1.;
const double well::MIN_redExp = 1.;
const double well::MAX_redExp = 3.05;
const double well::STP_redExp = 0.1;

well::well() { clear(); }                         // --- (constructor) ---

void well::clear() {                                      // --- clear ---
  index = 1; caWell = false; isReactant = false;
  connect.clear();
  fname.erase(); title.erase();
  offset = 0; lowest = 0; actchan = 0; sigma = 0.;
  press.clear(); temp.clear();
  thRhoGrad = DEF_thRhoGrad;
  redExpFix = DEF_redExpFix;
  rho.clear(); rate.clear();
}

int well::readFromUnimol() {                     // --- readFromUnimol ---
  int ich, i, inc, nalpha, ixv, jxv, np, jav, ntemp, ioptht, ioptpr;
  double err1, err2, e0, pr, palmt, tv, wt1, wt2, rhv, corrat, rtv;
  string basn = getBasePart(fname), extn = getExtPart(fname);
  ifstream inf;
  vector<double> rtvl;

  if (fname == "") {
    cout << "No file name is given for well-" << index << ".\n";
    return ERRE;
  }
  if (extn == "") { fname = basn + ".dat"; }
  inf.open(fname.c_str());
  if (!inf)
    { cout << "failed to open [" << fname << "].\n"; return ERRE; }
  if (getLineTrim(inf, title) == ERRE) { return ERRE; }
  inf >> inc >> nchan >> grsize;
  inf >> err1 >> err2 >> err3;
  inf >> e0;
  inf >> nalpha;
  if (nalpha < 0) { alpTex = abs(nalpha) / 2.; }
  else { alpTex = 0.; }
  inf >> alpha1000;
  inf >> ixv >> jxv;
  inf >> np;
  for (i = 0; i < np; i++) { inf >> pr; press.push_back(pr); }
  press.sortElimDup();
  inf >> palmt;
  inf >> jav;
  inf >> ntemp;
  for (i = 0; i < ntemp; i++) { inf >> tv; temp.push_back(tv); }
  temp.sortElimDup();
  inf >> sigma >> wt1 >> wt2 >> eps;
  redmass = wt1 * wt2 / (wt1 + wt2);
  inf >> ioptht >> ioptpr;
  inf >> ngrain;
  for (i = 0; i < ngrain; i++) { inf >> rhv; rho.push_back(rhv); }
  for (i = 0; i < ntemp; i++) { inf >> tv >> corrat; }
  inf >> nreacg;
  for (ich = 0; ich < nchan; ich++) {
    rtvl.clear();
    for (i = 0; i < nreacg; i++) { inf >> rtv; rtvl.push_back(rtv); }
    rate.push_back(rtvl);
  }
  nnonrg = ngrain - nreacg;
  upbound = ngrain;
  err3 *= 1e-3;
  inf.close();
  return NORM;
}

void well::printParETP(ostream &os, int vb) {       // --- printParETP ---
  if (vb > 1) {
    os << "well-" << index << ": thRhoGrad = " << thRhoGrad;
    if (fabs(thRhoGrad - DEF_thRhoGrad) < 1.e-6) { os << " (default)"; }
    os << endl;
    os << "well-" << index << ": redExpon = ";
    if (redExpFix < 0.) { os << "auto (default)\n"; }
    else { os << redExpFix << endl; }
  }
  os << "well-" << index << ": err3 = " << err3 << endl;
}

void well::printParCA(ostream &os) {                 // --- printParCA ---
  if (caWell)
    { os << "well-" << index << ": recombChan = " << actchan + 1 << endl; }
  os << "well-" << index << ": truncate = " << lowest;
  if (lowest == 0) { os << " (no truncation)"; }
  os << endl;
}

int well::setCurrentTp(double Tin, double pin) {   // --- setCurrentTp ---
  double densM, vMean, crossSec, Omega22;
  if (sigma < 1.e-3) { cout << "well: data not read.\n"; return ERRE; }
  current.T = Tin; current.p = pin;
  current.alpha = alpha1000 * pow(current.T / 1000., alpTex);
  crossSec = M_PI * sigma * sigma * 1.e-16;            // cm2
  vMean = CVMEAN * sqrt(current.T / redmass);          // cm s-1
  densM = current.p / RTCMMLC / current.T;             // molecules cm-3
  Omega22 = 1. / (0.636 + 0.567 * log10(current.T / eps));
  current.collfreq = crossSec * vMean * densM * Omega22;
  return NORM;
}

/*------------------------------------------------------------------------
     List of Wells
------------------------------------------------------------------------*/

listWells::listWells() { clear(); }               // --- (constructor) ---

void listWells::clear() {                                 // --- clear ---
  totSize = 0; posReactWell = -1; topCut = 10;
  tableIndices.clear();
}

int listWells::isValidIndex(int idx) {             // --- isValidIndex ---
  if (tableIndices.find(idx) != tableIndices.end()) { return true; }
  return false;
}

int listWells::isUsedIndex(int idx)                 // --- isUsedIndex ---
  { return isValidIndex(idx); }

int listWells::autoIndex() {                          // --- autoIndex ---
  int idx = 0;
  do { idx++; } while (isUsedIndex(idx));
  return idx;
}

int listWells::getPos(int idx) {                         // --- getPos ---
  if (!isValidIndex(idx)) { return -1; }
  return tableIndices[idx];
}

int listWells::add(well &w) {                               // --- add ---
  int idx = w.index, pos = size();
  if (isUsedIndex(idx)) { return ERRE; }
  push_back(w);
  tableIndices.insert(make_pair(idx, pos));
  return NORM;
}

int listWells::readUnimol() {                        // --- readUnimol ---
  iterator p;
  for (p = begin(); p < end(); p++)
    { if (p->readFromUnimol() == ERRE) { return ERRE; } }
  if (checkConsistency() == ERRE) { return ERRE; }
  if (checkConnect() == ERRE) { return ERRE; }
  alignSize();
  if (checkEqkE() == ERRE) { return ERRE; }
  setChanInfo();
  return NORM;
}

void listWells::mergedTlist(tempList &tl) {         // --- mergedTlist ---
  iterator p;
  tl.clear();
  for (p = begin(); p < end(); p++) { tl.addList(p->temp); }
}

void listWells::mergedPlist(pressList &pl) {        // --- mergedPlist ---
  iterator p;
  pl.clear();
  for (p = begin(); p < end(); p++) { pl.addList(p->press); }
}

void listWells::printParETP(ostream &os, int vb) {  // --- printParETP ---
  iterator p;
  for (p = begin(); p < end(); p++) { p->printParETP(os, vb); }
}

void listWells::printParCA(ostream &os) {            // --- printParCA ---
  iterator p;
  for (p = begin(); p < end(); p++) { p->printParCA(os); }
}

int listWells::reduceSize(int maxs) {                // --- reduceSize ---
  int redWell;
  iterator p;
  if (totSize == 0) { alignSize(); }
  if (totSize <= maxs) { return NORM; }
  redWell = (totSize - maxs + 3) / size();
  absTop -= redWell;
  totSize = 0;
  for (p = begin(); p < end(); p++) {
    p->upbound = absTop - p->offset;
    totSize += p->upbound - p->lowest;
    if (p->upbound <= p->nnonrg) { return ERRE; }
  }
  return NORM;
}

void listWells::setPosBmat() {                       // --- setPosBmat ---
  int posB = 0;
  iterator p;
  for (p = begin(); p < end(); p++) {
    p->posBmat = posB;
    posB += p->upbound - p->lowest;
  }
}

void listWells::setErr3(double e) {                     // --- setErr3 ---
  iterator p;
  for (p = begin(); p < end(); p++) { p->err3 = e; }
}

int listWells::checkCA() {                              // --- checkCA ---
  int ncaWell = 0;
  iterator p;
  for (p = begin(); p < end(); p++) {
    if (p->caWell) { ncaWell++; }
  }
  if (ncaWell == 0)
    { cout << "No recombination channel.\n"; return ERRE; }
  else if (ncaWell > 1)
    { cout << "Too many recombination channels.\n"; return ERRE; }
  return NORM;
}

int listWells::checkConsistency() {            // --- checkConsistency ---
  int posw, grsz0, ncaw = 0;
  double redm0;
  if ((*this)[0].caWell) { ncaw++; }
  grsz0 = (*this)[0].grsize;
  redm0 = (*this)[0].redmass;
  for (posw = 1; posw < size(); posw++) {
    if ((*this)[posw].caWell) { ncaw++; }
    if ((*this)[posw].grsize != grsz0)
      { cout << "Grain size differs between wells.\n"; return ERRE; }
    if (fabs((*this)[posw].redmass - redm0) > 1.e-6)
      { cout << "Masses differ between wells.\n"; return ERRE; }
  }
  if (ncaw > 1)
    { cout << "More than one wells have recombChan.\n"; return ERRE; }
  return NORM;
}

int listWells::checkConnect() {                    // --- checkConnect ---
  int pos, ic, toIdx, toPos, chan, asym;
  vector<vector<int> > ncnx(size(), vector<int>(size(), 0));

  for (pos = 0; pos < size(); pos++) {
    for (ic = 0; ic < (*this)[pos].connect.size(); ic++) {
      if (!isValidIndex(toIdx = (*this)[pos].connect[ic].toWellIdx)) {
        cout << "well-" << (*this)[pos].index << ": Invalid connect-toWellIdx "
             << toIdx << endl;
        return ERRE;
      }
      toPos = tableIndices[toIdx];
      ncnx[pos][toPos]++;
      chan = (*this)[pos].connect[ic].chan;
      if ((chan < 0) || (chan >= (*this)[pos].nchan)) {
        cout << "well-" << (*this)[pos].index << ": Invalid connect-chan "
             << chan + 1 << endl;
        return ERRE;
      }
    }
  }
  asym = false;
  for (pos = 0; pos < size(); pos++) {
    for (toPos = pos + 1; toPos < size(); toPos++) {
      if (ncnx[toPos][pos] != ncnx[pos][toPos]) { asym = true; break; }
    }
    if (asym) { break; }
  }
  if (asym) {
    cout << "Asymmetric connection matrix.\n";
    for (pos = 0; pos < size(); pos++) {
      for (toPos = 0; toPos < size(); toPos++)
        { cout << ' ' << ncnx[pos][toPos]; }
      cout << endl;
    }
    return ERRE;
  }
  return NORM;
}

void listWells::alignSize() {                         // --- alignSize ---
  int pos, minOffset = (*this)[0].offset, top;

  // - set minimum offset to 0
  for (pos = 1; pos < size(); pos++) {
    if ((*this)[pos].offset < minOffset)
      { minOffset = (*this)[pos].offset; }
  }
  for (pos = 0; pos < size(); pos++) { (*this)[pos].offset -= minOffset; }
  // - align upbound & count total size
  absTop = (*this)[0].offset + (*this)[0].ngrain;
  for (pos = 1; pos < size(); pos++) {
    top = (*this)[pos].offset + (*this)[pos].ngrain;
    if (top < absTop) { absTop = top; }
  }
  absTop -= topCut;  // - top cut to avoid asymmetry problem
  totSize = 0;
  for (pos = 0; pos < size(); pos++) {
    (*this)[pos].upbound = absTop - (*this)[pos].offset;
    totSize += (*this)[pos].upbound - (*this)[pos].lowest;
  }
  cout << "listWells::alignSize absTop = " << absTop << endl;
}

int listWells::checkEqkE() {                          // --- checkEqkE ---
  int pos, toPos, ic, ig, botGr, ighere, igto, irhere, irto, ich;
  double kforw, kback, rhohere, rhoto, diffpc, dpcww, maxdpc = 0., corf;
  vector<int> forwChan, backChan;

  for (pos = 0; pos < size(); pos++) {
    for (toPos = pos + 1; toPos < size(); toPos++) {
      forwChan.clear();
      for (ic = 0; ic < (*this)[pos].connect.size(); ic++) {
        if (getPos((*this)[pos].connect[ic].toWellIdx) == toPos)
          { forwChan.push_back((*this)[pos].connect[ic].chan); }
      }
      if (forwChan.empty()) { continue; }
      backChan.clear();
      for (ic = 0; ic < (*this)[toPos].connect.size(); ic++) {
        if (getPos((*this)[toPos].connect[ic].toWellIdx) == pos)
          { backChan.push_back((*this)[toPos].connect[ic].chan); }
      }
      botGr = (*this)[pos].offset + (*this)[pos].nnonrg;
      if (((*this)[toPos].offset + (*this)[toPos].nnonrg) < botGr)
        { botGr = (*this)[toPos].offset + (*this)[toPos].nnonrg; }
      dpcww = 0.;
      for (ig = botGr; ig < absTop; ig++) {
        ighere = ig - (*this)[pos].offset;
        irhere = ighere - (*this)[pos].nnonrg;
        igto = ig - (*this)[toPos].offset;
        irto = igto - (*this)[toPos].nnonrg;
        if ((ighere < 0) || (igto < 0)) { continue; }
        kforw = 0.;
        if (irhere >= 0) {
          for (ich = 0; ich < forwChan.size(); ich++)
            { kforw += (*this)[pos].rate[forwChan[ich]][irhere]; }
        }
        kback = 0.;
        if (irto >= 0) {
          for (ich = 0; ich < backChan.size(); ich++)
            { kback += (*this)[toPos].rate[backChan[ich]][irto]; }
        }
        rhohere = (*this)[pos].rho[ighere];
        rhoto = (*this)[toPos].rho[igto];
        diffpc = 0.;
        if (fabs(kback) <= 1.e-64) {
          if (fabs(kforw) > 1.e-64) {
            for (ich = 0; ich < forwChan.size(); ich++)
              { (*this)[pos].rate[forwChan[ich]][irhere] = 0.; }
          }
        } else {
          diffpc = fabs((kforw / kback) / (rhoto / rhohere) - 1.) * 100.;
          if (diffpc > 1.) {
            corf = sqrt((rhoto / rhohere) / (kforw / kback));
            for (ich = 0; ich < forwChan.size(); ich++)
              { (*this)[pos].rate[forwChan[ich]][irhere] *= corf; }
            for (ich = 0; ich < backChan.size(); ich++)
              { (*this)[toPos].rate[backChan[ich]][irto] /= corf; }
          }
        }
        if (diffpc > dpcww) { dpcww = diffpc; }
      }
      if (dpcww > maxdpc) { maxdpc = dpcww; }
      cout << "max asymmetry [%] between well" << (*this)[pos].index
           << " and well" << (*this)[toPos].index << " = " << dpcww
           << endl;
    }
  }
  if (maxdpc > 20.) {
    cout << "ERROR: large k(E) max asymmetry [%] = " << maxdpc << endl;
    return ERRE;
  }
  if (maxdpc > 2.)
    { cout << "WARNING: k(E) max asymmetry [%] = " << maxdpc << endl; }
  return NORM;
}

void listWells::setChanInfo() {                     // --- setChanInfo ---
  int posw, ch, icn;
  nTotCh = 0; nTotOCh = 0;
  for (posw = 0; posw < size(); posw++) {
    (*this)[posw].chInf.clear();
    for (ch = 0; ch < (*this)[posw].nchan; ch++)
      { (*this)[posw].chInf.push_back(-1); }
    for (icn = 0; icn < (*this)[posw].connect.size(); icn++) {
      (*this)[posw].chInf[(*this)[posw].connect[icn].chan]
       = this->getPos((*this)[posw].connect[icn].toWellIdx);
    }
    (*this)[posw].nOutCh = 0;
    for (ch = 0; ch < (*this)[posw].nchan; ch++) {
      if ((*this)[posw].chInf[ch] == -1) { (*this)[posw].nOutCh++; }
    }
    nTotOCh += (*this)[posw].nOutCh;
    nTotCh += (*this)[posw].nchan;
  }
}

/*------------------------------------------------------------------------
     Unimolecular Problem Solver
------------------------------------------------------------------------*/

/*----- public member functions ----------------------------------------*/

umolProb::umolProb() { clear(); }                 // --- (constructor) ---

void umolProb::clear() {                                  // --- clear ---
  probtype = Eigen;
  maxCycLinSs = 300; maxCycItrSs = 1000;
  verbose = 0;
  psiz = 0;
  T = 0.; p = 0.;
}

void umolProb::init1W(well &w) {                         // --- init1W ---
  if ((w.upbound - w.lowest) > MXSIZ) {
    w.upbound = w.lowest + MXSIZ;
    cout << "umolProb::init1W upbound truncated to " << w.upbound << ".\n";
  }
  psiz = w.upbound - w.lowest;
  cout << "umolProb::init1W psiz = " << psiz << endl;
}

int umolProb::initMW(listWells &lw) {                    // --- initMW ---
  if (lw.totSize > MXSIZ) {
    if (lw.reduceSize(MXSIZ) == ERRE) {
      cout << "umolProb::initMW too small size available.\n";
      return ERRE;
    }
    cout << "umolProb::initMW size reduced to " << lw.totSize << ".\n";
  }
  psiz = lw.totSize;
  cout << "umolProb::initMW psiz = " << psiz << endl;
  lw.setPosBmat();
  return NORM;
}

void umolProb::calcHPL1W(well &w, double Tin) {       // --- calcHPL1W ---
  double gnorm;
  T = Tin; p = -1.;
  setBoltzRflx1W(w, Tin, 0);
  setSmat(w, 0);
  applyStoV();
  gnorm = normG1W(w, 0, v);
  integkEg1W(w, 0, v, gnorm);
}

void umolProb::calcHPLMW(listWells &lw, double Tin) { // --- calcHPLMW ---
  int posw;
  double tgnorm;
  T = Tin; p = -1.;
  setBoltzRflxMWall(lw, Tin);
  for (posw = 0; posw < lw.size(); posw++)
    { setSmat(lw[posw], lw[posw].posBmat); }
  applyStoV();
  tgnorm = normGMW(lw, v);
  integkEgMW(lw, v, tgnorm);
}

                                              // --- setBoltzRflxMWall ---
void umolProb::setBoltzRflxMWall(listWells &lw, double Tin) {
  int posw, shift, i;
  double gamma = exp(-lw[0].grsize / KB_WNK / Tin), tnorm = 0.;
  for (posw = 0; posw < lw.size(); posw++) {
    shift = lw[posw].posBmat - lw[posw].lowest;
    for (i = lw[posw].lowest; i < lw[posw].upbound; i++) {
      tnorm +=
       (v[i + shift] = lw[posw].rho[i] * pow(gamma, i + lw[posw].offset));
    }
  }
  for (i = 0; i < psiz; i++) { v[i] /= tnorm; }
}

                                              // --- setBoltzRflxMWsel ---
void umolProb::setBoltzRflxMWsel(listWells &lw, double Tin) {
  int poswr = lw.posReactWell;
  setBoltzRflx1W(lw[poswr], Tin, lw[poswr].posBmat);
}

                                                 // --- setBoltzRflx1W ---
void umolProb::setBoltzRflx1W(well &w, double Tin, int pos0) {
  int i, shift = pos0 - w.lowest;
  double gamma = exp(-w.grsize / KB_WNK / Tin), norm = 0.;
  for (i = w.lowest; i < w.upbound; i++)
    { norm += (v[i + shift] = w.rho[i] * pow(gamma, i)); }
  for (i = w.lowest; i < w.upbound; i++)
    { v[i + shift] /= norm; }
}

                                                   // --- driveLinSs1W ---
int umolProb::driveLinSs1W(well &w, double Tin, double pin) {
  int lcnt = 0, dcnt = 0;
  double knew = 0., kold, dknew = 0., dkold, step = 0.5;
  do {
    kold = knew; dkold = dknew;
    if (setMat1W(w, Tin, pin) == ERRE) { return ERRE; }
    if (solveLinEq1W(w) == ERRE) { return ERRE; }
    lcnt++; dcnt++;
    knew = w.current.ktot; dknew = knew - kold;
    if (dcnt > 3) {
      dcnt = 0;
      if (dknew * dkold < 0) { step *= 0.5; }
      else {
        step *= 1.5;
        if (step > 1.) { step = 1.; }
      }
    }
    storeRflx(w, 0, step);
    cout << "umolProb::driveLinSs1W ===== LOOP " << lcnt << " ====="
         << " step = " << step
         << ", k = " << knew << endl;
    if (lcnt >= maxCycLinSs) {
      cout << "umolProb::driveLinSs1W *** MAXCYCLE ***\n";
      return ERRE;
    }
  } while (fabs((kold - knew) / knew) > 1.e-5);
  return NORM;
}

                                                   // --- driveItrSs1W ---
int umolProb::driveItrSs1W(well &w, double Tin, double pin) {
  int lcnt = 0, dcnt = 0;
  double gnorm, kold, knew, dgnabs, step = 0.05, dknew = 0., dkold;
  if (setMat1W(w, Tin, pin) == ERRE) { return ERRE; }
  gnorm = normG1W(w, 0, r[0]);
  integkEg1W(w, 0, r[0], gnorm);
  knew = w.current.ktot;
  do {
    kold = knew; dkold = dknew;
    dgnabs = calcDGdt();
    addDGstep(dgnabs, step);
    gnorm = normG1W(w, 0, r[0]);
    integkEg1W(w, 0, r[0], gnorm);
    knew = w.current.ktot; dknew = knew - kold;
    lcnt++; dcnt++;
    if (dcnt > 3) {
      dcnt = 0;
      if (dknew * dkold < 0) { step *= 0.8; }
      else {
        step *= 1.2;
        if (step > 0.6) { step = 0.6; }
      }
    }
    cout << "umolProb::driveItrSs1W ===== LOOP " << lcnt << " ====="
         << " step = " << step
         << ", k = " << knew << ' ' << kold << endl;
    if (lcnt >= maxCycItrSs) {
      cout << "umolProb::driveItrSs1W *** MAXCYCLE ***\n";
      return ERRE;
    }
  } while (fabs((kold - knew) / knew) > 1.e-5);
  storeSSG();
  return NORM;
}

                                                   // --- driveLinSsMW ---
int umolProb::driveLinSsMW(listWells &lw, double Tin, double pin) {
  int lcnt = 0, dcnt = 0, poswr = lw.posReactWell, posB;
  if (poswr >= 0) { posB = lw[poswr].posBmat; }
  double knew = 0., kold, dknew = 0., dkold, step = 0.5;
  do {
    kold = knew; dkold = dknew;
    if (setMatMW(lw, Tin, pin) == ERRE) { return ERRE; }
    if (solveLinEqMW(lw) == ERRE) { return ERRE; }
    lcnt++; dcnt++;
    knew = lw.koutTot; dknew = knew - kold;
    // if (knew < 0.) { step *= 0.1; dcnt = 0; }
    if (dcnt > 3) {
      dcnt = 0;
      if (dknew * dkold < 0) { step *= 0.5; }
      else {
        step *= 1.5;
        if (step > 1.) { step = 1.; }
      }
    }
    if (poswr < 0) { storeRflxAll(step); }
    else { storeRflx(lw[poswr], posB, step); }
    cout << "umolProb::driveLinSsMW ===== LOOP " << lcnt << " ====="
         << " step = " << step
         << ", k = " << knew << endl;
    if (lcnt >= maxCycLinSs) {
      cout << "umolProb::driveLinSsMW *** MAXCYCLE ***\n";
      return ITRERR;
    }
  } while (fabs((kold - knew) / knew) > 1.e-5);
  return NORM;
}

                                                   // --- driveItrSsMW ---
int umolProb::driveItrSsMW(listWells &lw, double Tin, double pin) {
  int lcnt = 0, dcnt = 0;
  double tgnorm, kold, knew, dgnabs, step = 0.05, dknew = 0., dkold;
  if (setMatMW(lw, Tin, pin) == ERRE) { return ERRE; }
  tgnorm = normGMW(lw, r[0]);
  integkEgMW(lw, r[0], tgnorm);
  knew = lw.koutTot;
  do {
    kold = knew; dkold = dknew;
    dgnabs = calcDGdt();
    addDGstep(dgnabs, step);
    tgnorm = normGMW(lw, r[0]);
    integkEgMW(lw, r[0], tgnorm);
    knew = lw.koutTot; dknew = knew - kold;
    lcnt++; dcnt++;
    if (dcnt > 3) {
      dcnt = 0;
      if (dknew * dkold < 0) { step *= 0.8; }
      else {
        step *= 1.2;
        if (step > 0.6) { step = 0.6; }
      }
    }
    cout << "umolProb::driveItrSsMW ===== LOOP " << lcnt << " ====="
         << " step = " << step
         << ", k = " << knew << ' ' << kold << endl;
    if (lcnt >= maxCycItrSs) {
      cout << "umolProb::driveItrSsMW *** MAXCYCLE ***\n";
      return ERRE;
    }
  } while (fabs((kold - knew) / knew) > 1.e-5);
  storeSSG();
  return NORM;
}

                                                       // --- setMat1W ---
int umolProb::setMat1W(well &w, double Tin, double pin) {
  T = Tin; p = pin;
  if (drvSetETP(w, 0, noffdiag) == ERRE) { return ERRE; }
  setRateSmat(w, 0);
  if (probtype == LinEq) { clearRflx(); setRflxChan(w, 0); }
  if (probtype == LinSS) { restoreRflx(w, 0); }
  if (probtype == ItrSS) { restoreSSG(); }
  if (symmetMat() == ERRE) { return ERRE; }
  makeMatBanded(w, 0);
  return NORM;
}

                                                       // --- setMatMW ---
int umolProb::setMatMW(listWells &lw, double Tin, double pin) {
  int noffd, pwell, posB;
  T = Tin; p = pin;
  noffdiag = 0;
  if ((probtype == LinEq) || (probtype == LinSS)) { clearRflx(); }
  for (pwell = 0; pwell < lw.size(); pwell++) {
    posB = lw[pwell].posBmat;
    if (drvSetETP(lw[pwell], posB, noffd) == ERRE) { return ERRE; }
    if (noffd > noffdiag) { noffdiag = noffd; }
    setRateSmat(lw[pwell], posB);
    if ((probtype == LinEq) && lw[pwell].caWell)
      { setRflxChan(lw[pwell], posB); }
    if ((probtype == LinSS) && lw[pwell].isReactant)
      { restoreRflx(lw[pwell], posB); }
  }
  if ((probtype == ItrSS) || ((probtype == LinSS) && (lw.posReactWell < 0)))
    { restoreSSG(); }
  setCrossRate(lw);
  if (symmetMat() == ERRE) { return ERRE; }
  for (pwell = 0; pwell < lw.size(); pwell++) {
    posB = lw[pwell].posBmat;
    makeMatBanded(lw[pwell], posB);
  }
  return NORM;
}

int umolProb::solveEigen1W(well &w) {              // --- solveEigen1W ---
  int info;
  double gnorm;
  if (verbose > 0) { cout << " Calling lapack_dsyev ...\n"; }
  info = lapack_dsyev('V', 'U', psiz, (double **)b, MXSIZ, v, work, WMSIZ);
  if (info != 0) {
    cout << "umolProb::solveEigen1W lapack_dsyev returnd info = " << info
         << ".\n";
    return ERRE;
  }
  esiz = psiz; findLargeNegEigen();
  gnorm = normG1W(w, 0, b[posLNE]);
  integkEg1W(w, 0, b[posLNE], gnorm);
  return NORM;
}

/*
int umolProb::solveEigenBand1W(well &w) {      // --- solveEigenBand1W ---
  int info;
  double gnorm;
  if (noffdiag + 1 > BDSIZ) { return solveEigen1W(w); }
  prepABmat();
  if (verbose > 0) {
    cout << " Calling lapack_dsbev (for banded, noffdiag=" << noffdiag
         << ") ...\n";
  }
  info = lapack_dsbev('V', 'U', psiz, noffdiag, (double **)ab, BDSIZ, v,
                      (double **)b, MXSIZ, work);
  if (info != 0) {
    cout << "umolProb::solveEigenBand1W lapack_dsbev returnd info = "
         << info << ".\n";
    return ERRE;
  }
  esiz = psiz; findLargeNegEigen();
  gnorm = normG1W(w, 0, b[posLNE]);
  integkEg1W(w, 0, b[posLNE], gnorm);
  return NORM;
}
*/

int umolProb::solveEigen1Wdsyevr(well &w) {  // --- solveEigen1Wdsyevr ---
  int info, posmin;
  double gnorm;
  posmin = psiz * 0.8;
  if (verbose > 0) { cout << " Calling lapack_dsyevr ...\n"; }
  info = lapack_dsyevr('V', 'I', 'U', psiz, (double **)b, MXSIZ, 0., 0.,
   posmin, psiz, 0., esiz, v, (double **)z, MXSIZ, isuppz, work, WMSIZ,
   iwork, WISIZ);
  if (verbose > 0) {
    cout << "posmin = " << posmin << ", esiz = " << esiz << endl;
    cout << "lapack DSYEVR: ";
    cout << "optimal WMSIZ = " << work[0];
    cout << ", optimal WISIZ = " << iwork[0] << endl;
  }
  if (info != 0) {
    cout << "umolProb::solveEigen1Wdsyevr lapack_dsyevr returnd info = "
         << info << ".\n";
    return ERRE;
  }
  findLargeNegEigen();
  gnorm = normG1W(w, 0, z[posLNE]);
  integkEg1W(w, 0, z[posLNE], gnorm);
  return NORM;
}

int umolProb::solveLinEq1W(well &w) {              // --- solveLinEq1W ---
  int info;
  double gnorm;
  if (verbose > 0) { cout << " Calling lapack_dsysv ...\n"; }
  info = lapack_dsysv('U', psiz, 1, (double **)b, MXSIZ, ipiv, (double **)r,
                      MXSIZ, work, WMSIZ);
  if (info != 0) {
    cout << "umolProb::solveLinEq1W lapack_dsysv returnd info = " << info
         << ".\n";
    return ERRE;
  }
  gnorm = normG1W(w, 0, r[0]);
  integkEg1W(w, 0, r[0], gnorm);
  return NORM;
}
                                                  //--- outputVector1W ---
void umolProb::outputVector1W(ostream &os, well &w) {
  int i, imat;
  double *g, gnorm;
  if (probtype == Eigen) { g = b[posLNE]; }
  else { g = r[0]; }
  gnorm = normG1W(w, 0, g);
  os << "T[K],P[Torr]\n";
  os << w.current.T << ',' << w.current.p << endl;
  os << "grain#,E[cm-1],g\n";
  for (i = 0; i < w.upbound; i++) {
    os << i << ',' << i * w.grsize << ',';
    if (i < w.lowest) {
      os << "0\n";
    } else {
      imat = i - w.lowest;
      os << g[imat] * s[imat] / gnorm << endl;
    }
  }
}

int umolProb::solveEigenMW(listWells &lw) {        // --- solveEigenMW ---
  int info, goodEigen, posw, ik;
  double tgnorm;
  if (verbose > 0) { cout << " Calling lapack_dsyev ...\n"; }
  info = lapack_dsyev('V', 'U', psiz, (double **)b, MXSIZ, v, work, WMSIZ);
  if (info != 0) {
    cout << "umolProb::solveEigenMW lapack_dsyev returnd info = " << info
         << ".\n";
    return ERRE;
  }
  esiz = psiz; findLargeNegEigen();
  if (lw.posReactWell >= 0) {
    do {
      goodEigen = true;
      tgnorm = normGMW(lw, b[posLNE]);
      for (posw = 0; posw < lw.size(); posw++)
        { if (lw[posw].current.fpop < 0.) { goodEigen = false; } }
      if (lw[lw.posReactWell].current.fpop < 0.5) { goodEigen = false; }
      if (!goodEigen) {
        if (posLNE == 0) {
          cout << "umolProb::solveEigenMW no good eigen value was found.\n";
          return ERRE;
        } else {
          posLNE--;
        }
      }
    } while (!goodEigen);
  } else {
    tgnorm = normGMW(lw, b[posLNE]);
  }
  integkEgMW(lw, b[posLNE], tgnorm);
  return NORM;
}

                                             // --- solveEigenMWdsyevr ---
int umolProb::solveEigenMWdsyevr(listWells &lw) {
  int info, goodEigen, posw, ik, posmin;
  double tgnorm;
  posmin = psiz * 0.8;
  if (verbose > 0) { cout << " Calling lapack_dsyevr ...\n"; }
  info = lapack_dsyevr('V', 'I', 'U', psiz, (double **)b, MXSIZ, 0., 0.,
   posmin, psiz, 0., esiz, v, (double **)z, MXSIZ, isuppz, work, WMSIZ,
   iwork, WISIZ);
  if (verbose > 0) {
    cout << "posmin = " << posmin << ", esiz = " << esiz << endl;
    cout << "lapack DSYEVR: ";
    cout << "optimal WMSIZ = " << work[0];
    cout << ", optimal WISIZ = " << iwork[0] << endl;
  }
  if (info != 0) {
    cout << "umolProb::solveEigenMWdsyevr lapack_dsyevr returnd info = "
         << info << ".\n";
    return ERRE;
  }
  findLargeNegEigen();
  if (lw.posReactWell >= 0) {
    do {
      goodEigen = true;
      tgnorm = normGMW(lw, z[posLNE]);
      for (posw = 0; posw < lw.size(); posw++)
        { if (lw[posw].current.fpop < 0.) { goodEigen = false; } }
      if (lw[lw.posReactWell].current.fpop < 0.5) { goodEigen = false; }
      if (!goodEigen) {
        if (posLNE == 0) {
          cout << "umolProb::solveEigenMW no good eigen value was found.\n";
          return ERRE;
        } else {
          posLNE--;
        }
      }
    } while (!goodEigen);
  } else {
    tgnorm = normGMW(lw, z[posLNE]);
  }
  integkEgMW(lw, z[posLNE], tgnorm);
  return NORM;
}

int umolProb::solveLinEqMW(listWells &lw) {        // --- solveLinEqMW ---
  int info;
  double tgnorm;
  if (verbose > 0) { cout << " Calling lapack_dsysv ...\n"; }
  info = lapack_dsysv('U', psiz, 1, (double **)b, MXSIZ, ipiv, (double **)r,
                      MXSIZ, work, WMSIZ);
  if (info != 0) {
    cout << "umolProb::solveLinEqMW lapack_dsysv returnd info = " << info
         << ".\n";
    return ERRE;
  }
  tgnorm = normGMW(lw, r[0]);
  integkEgMW(lw, r[0], tgnorm);
  return NORM;
}

                                                 // --- outputVectorMW ---
void umolProb::outputVectorMW(ostream &os, listWells &lw) {
  int posw, i, iw, im, pos0;
  double *g, tgnorm, gsiz = lw[0].grsize;
  if (probtype == Eigen) { g = b[posLNE]; }
  else { g = r[0]; }
  tgnorm = normGMW(lw, g);

  os << "T[K],P[Torr]\n";
  os << T << ',' << p << endl;
  os << "grain#,E[cm-1]";
  for (posw = 0; posw < lw.size(); posw++) { os << ",g" << lw[posw].index; }
  os << endl;
  for (i = 0; i < lw.absTop; i++) {
    os << i << ',' << i * gsiz;
    for (posw = 0; posw < lw.size(); posw++) {
      os << ',';
      iw = i - lw[posw].offset;
      im = iw - lw[posw].lowest;
      pos0 = lw[posw].posBmat;
      if (iw >= 0) {
        if (im < 0) { os << '0'; }
        else { os << g[pos0 + im] * s[pos0 + im] / tgnorm; }
      }
    }
    os << endl;
  }
}

                                                       // --- printMat ---
void umolProb::printMat(ostream &os, int start, double fact) {
  int i, j, end = start + 255;
  if (end > psiz) { end = psiz; }
  os << "umolProb::printMat\n";
  for (i = start; i < end; i++) {
    for (j = start; j < end; j++) { os << b[j][i] * fact << ','; }
    os << endl;
  }
}

void umolProb::outEigVal(ostream &os, int n) {        // --- outEigVal ---
  // - print out largest n eigenvalues
  int i, i0;
  i0 = esiz - n;
  if (i0 < 0) { i0 = 0; }
  for (i = i0; i < esiz; i++) {
    os << v[i];
    if (i < esiz - 1) { os << ','; }
    else { os << endl; }
  }
}

/*----- protected member functions -------------------------------------*/

                                                      // --- drvSetETP ---
int umolProb::drvSetETP(well &w, int pos0, int &noff) {
  int retSETP;
  if (w.setCurrentTp(T, p) == ERRE) { return ERRE; }
  if (w.redExpFix < 0.) {                       // --- auto
    w.current.redExpon = w.MIN_redExp;
    do {
      if ((retSETP = setETProb(w, pos0, noff)) == ERRE) {
        w.current.redExpon += w.STP_redExp;
        if (w.current.redExpon > w.MAX_redExp) {
          cout << "Exceeded MAX-redExpon. Abort.\n";
          break;
        } else {
          if (verbose > 1) {
            cout << "  Retrying with redExpon=" << w.current.redExpon << endl;
          }
        }
      }
    } while (retSETP == ERRE);
    if (retSETP == ERRE) { return ERRE; }
  } else {                                      // --- fixed
    w.current.redExpon = w.redExpFix;
    if (setETProb(w, pos0, noff) == ERRE) { return ERRE; }
  }
  return NORM;
}

                                                      // --- setETprob ---
int umolProb::setETProb(well &w, int pos0, int &noff) {
  int jc, ir, shift = pos0 - w.lowest, upref, ncutlow, ncut;
  double alpha, z, beta, gamma, Ke, pup, pdown, dnorm, usum, pupden;

  alpha = w.current.alpha; z = w.current.collfreq;
  beta = exp(-w.grsize / alpha);
  gamma = exp(-w.grsize / KB_WNK / T);
  upref = (int)(alpha * 1.5 / w.grsize) + 1;
  noff = (int)(log(w.err3) / log(beta)) + 1;
  ncutlow = 0;
  for (ir = w.ngrain - upref - 1; ir >= 0; ir--) {
    if (w.rho[ir + upref] / w.rho[ir] > w.thRhoGrad)
      { ncutlow = ir + 1; break; }
  }
  double *redfac = new double[ncutlow];
  for (ir = 0; ir < ncutlow; ir++) {
    redfac[ir] = pow(w.rho[ir + upref] / w.rho[ir], w.current.redExpon);
  }
// --- for debugging
//  cout << "umolProb::setETProb ncutlow=" << ncutlow << endl;
//
  double *norm = new double[w.ngrain];
  for (jc = w.ngrain - 1; jc >= w.lowest; jc--) {
    ncut = ncutlow;
    if (jc < ncutlow) { ncut = jc; }
    dnorm = (1. - pow(beta, jc - ncut + 1)) / (1. - beta);
    for (ir = ncut - 1; ir >=0; ir--)
      { dnorm += pow(beta, jc - ir) / redfac[ir]; }
    usum = 0.;
    pupden = 1.;
    if (jc < ncutlow) { pupden = redfac[jc]; }
    for (ir = jc + 1; ir < w.ngrain; ir++) {
      Ke = w.rho[ir] / w.rho[jc] * pow(gamma, ir - jc);
      pup = pow(beta, ir - jc) / pupden * Ke / norm[ir];
      usum += pup;
      if (ir < w.upbound) { b[jc + shift][ir + shift] = z * pup; }
    }
    if (usum >= 1.) {
      if (verbose > 1) {
      cout << "  umolProb::setETProb problem detected with thRhoGrad="
           << w.thRhoGrad << " and redExpon=" << w.current.redExpon << endl;
      }
      delete[] norm, redfac;
      return ERRE;
    }
    norm[jc] = dnorm / (1. - usum);
    for (ir = jc; ir >= w.lowest; ir--) {
      pdown = pow(beta, jc - ir) / norm[jc];
      if (ir < ncut) { pdown /= redfac[ir]; }
      if (ir == jc) { b[jc + shift][ir + shift] = z * (pdown - 1); }
      else { b[jc + shift][ir + shift] = z * pdown; }
    }
    // store sum of the downward transfer rate below w.lowest in d[]
    d[jc + shift] = 0.;
    for (ir = w.lowest - 1; ir >= 0; ir--) {
      pdown = pow(beta, jc - ir) / norm[jc];
      if (ir < ncut) { pdown /= redfac[ir]; }
      d[jc + shift] += z * pdown;
    }
// --- for debugging
//    cout << norm[jc] << ',' << usum << ',' << w.rho[jc] << endl;
//
  }
  delete[] norm, redfac;
  return NORM;
}

void umolProb::setRateSmat(well &w, int pos0) {     // --- setRateSmat ---
  setRate(w, pos0); setSmat(w, pos0);
}

void umolProb::setRate(well &w, int pos0) {             // --- setRate ---
  int i, irat, ch, shift = pos0 - w.lowest;
  double ktot;
  for (i = w.lowest; i < w.upbound; i++) {
    ktot = 0.;
    if ((irat = i - w.nnonrg) >= 0) {
      for (ch = 0; ch < w.nchan; ch++) { ktot += w.rate[ch][irat]; }
    }
    b[i + shift][i + shift] -= ktot;
  }
}

void umolProb::setSmat(well &w, int pos0) {             // --- setSmat ---
  int i, shift = pos0 - w.lowest;
  double gamma = exp(-w.grsize / KB_WNK / T), f;
  for (i = w.lowest; i < w.upbound; i++) {
    f = w.rho[i] * pow(gamma, i + w.offset);
    s[i + shift] = sqrt(f);
  }
}

void umolProb::setCrossRate(listWells &lw) {       // --- setCrossRate ---
  int pfr, pto, mfr, mto, mfr0, mto0, mfrL, mtoL, icn;
  int gfr, gto, rfr, rto;
  double kf, kb;
  vector<int> fwChan, bkChan;
  for (pfr = 0; pfr < lw.size(); pfr++) {
    for (pto = pfr + 1; pto < lw.size(); pto++) {
      mfr0 = lw[pfr].posBmat; mto0 = lw[pto].posBmat;
      mfrL = mfr0 + lw[pfr].upbound - lw[pfr].lowest;
      mtoL = mto0 + lw[pto].upbound - lw[pto].lowest;
      for (mfr = mfr0; mfr < mfrL; mfr++) {
        for (mto = mto0; mto < mtoL; mto++)
          { b[mfr][mto] = 0.; b[mto][mfr] = 0.; }
      }
      fwChan.clear();
      for (icn = 0; icn < lw[pfr].connect.size(); icn++) {
        if (lw.getPos(lw[pfr].connect[icn].toWellIdx) == pto)
          { fwChan.push_back(lw[pfr].connect[icn].chan); }
      }
      if (fwChan.empty()) { continue; }
      bkChan.clear();
      for (icn = 0; icn < lw[pto].connect.size(); icn++) {
        if (lw.getPos(lw[pto].connect[icn].toWellIdx) == pfr)
          { bkChan.push_back(lw[pto].connect[icn].chan); }
      }
      for (gfr = lw[pfr].lowest; gfr < lw[pfr].upbound; gfr++) {
        rfr = gfr - lw[pfr].nnonrg;
        kf = 0.;
        if (rfr >= 0) {
          for (icn = 0; icn < fwChan.size(); icn++)
            { kf += lw[pfr].rate[fwChan[icn]][rfr]; }
        }
        gto = gfr + lw[pfr].offset - lw[pto].offset;
        rto = gto - lw[pto].nnonrg;
        kb = 0.;
        if (rto >= 0) {
          for (icn = 0; icn < bkChan.size(); icn++)
            { kb += lw[pto].rate[bkChan[icn]][rto]; }
        }
        if (gto >= lw[pto].lowest) {
          mfr = mfr0 + gfr - lw[pfr].lowest;
          mto = mto0 + gto - lw[pto].lowest;
          b[mfr][mto] = kf; b[mto][mfr] = kb;
        }
      }
    }
  }
}

void umolProb::clearRflx() {                          // --- clearRflx ---
  int i;
  for (i = 0; i < psiz; i++) { r[0][i] = 0.; }
}

void umolProb::setRflxChan(well &w, int pos0) {     // --- setRflxChan ---
  int i, shift = pos0 - w.lowest, irat;
  double rf, sumrf = 0., gamma = exp(-w.grsize / KB_WNK / T);
  for (i = w.lowest; i < w.upbound; i++) {
    irat = i - w.nnonrg;
    if (irat < 0) { r[0][i + shift] = 0.; }
    else {
      rf = w.rho[i] * pow(gamma, i) * w.rate[w.actchan][irat];
      r[0][i + shift] = rf; sumrf += rf;
    }
  }
  for (i = w.lowest; i < w.upbound; i++) { r[0][i + shift] /= (-sumrf); }
}

void umolProb::restoreRflx(well &w, int pos0) {     // --- restoreRflx ---
  int i, shift = pos0 - w.lowest;
  for (i = w.lowest; i < w.upbound; i++) { r[0][i + shift] = v[i + shift]; }
}

                                                      // --- storeRflx ---
void umolProb::storeRflx(well &w, int pos0, double stp) {
  int i, shift = pos0 - w.lowest;
  double norm = 0.;
  for (i = w.lowest; i < w.upbound; i++)
    { norm += r[0][i + shift] * s[i + shift]; }
  for (i = w.lowest; i < w.upbound; i++) {
    v[i + shift] *= (1. - stp);
    v[i + shift] += (r[0][i + shift] * s[i + shift] / norm) * stp;
  }
}

void umolProb::storeRflxAll(double stp) {          // --- storeRflxAll ---
  int i;
  double norm = 0.;
  for (i = 0; i < psiz; i++) { norm += r[0][i] * s[i]; }
  for (i = 0; i < psiz; i++) {
    v[i] *= (1. - stp);
    v[i] += (r[0][i] * s[i] / norm * stp);
  }
}

void umolProb::restoreSSG() {                        // --- restoreSSG ---
  int i;
  for (i = 0; i < psiz; i++) { r[0][i] = v[i]; }
}

void umolProb::storeSSG() {                            // --- storeSSG ---
  int i;
  double norm = 0.;
  for (i = 0; i < psiz; i++) {
    norm += (v[i] = r[0][i] * s[i]);
  }
  for (i = 0; i < psiz; i++) { v[i] /= norm; }
}

int umolProb::symmetMat() {                           // --- symmetMat ---
  int ir, jc;
  double diffpc, maxdpc = 0.;
  for (ir = 0; ir < psiz; ir++) {
    for (jc = 0; jc < psiz; jc++) { b[jc][ir] *= s[jc] / s[ir]; }
  }
  for (ir = 0; ir < psiz - 1; ir++) {
    for (jc = ir + 1; jc < psiz; jc++) {
      if (b[jc][ir] < 0.)
        { cout << "Negative off-diagonal: [" << ir << "][" << jc << "]\n"; }
      else if (b[ir][jc] < 0.)
        { cout << "Negative off-diagonal: [" << ir << "][" << jc << "]\n"; }
      else if (b[jc][ir] * b[ir][jc] > 1.e-64) {
        diffpc = fabs(b[ir][jc] / b[jc][ir] - 1.) * 100.;
        if (diffpc > maxdpc) { maxdpc = diffpc; }
      }
    }
  }
  if (maxdpc > 5.) {
    cout << "ERROR: asymmetric b-mat - asymmetry [%] = " << maxdpc << endl;
    return ERRE;
  }
  if (maxdpc > 1.) {
    cout << "WARNING: b-mat max asymmetry [%] = " << maxdpc << endl;
  } else {
    if (verbose > 0)
      { cout << "b-mat max asymmetry [%] = " << maxdpc << endl; }
  }
  if ((probtype == LinEq) || (probtype == LinSS) || (probtype == ItrSS))
    // also apply S to r
    { applyStoR(); }
  return NORM;
}

void umolProb::applyStoR() {                          // --- applyStoR ---
  int i;
  for (i = 0; i < psiz; i++) { r[0][i] /= s[i]; }
}

void umolProb::applyStoV() {                          // --- applyStoV ---
  int i;
  for (i = 0; i < psiz; i++) { v[i] /= s[i]; }
}

void umolProb::makeMatBanded(well &w, int pos0) { // --- makeMatBanded ---
  int ir, ic, shift = pos0 - w.lowest;
  for (ir = w.lowest; ir < w.upbound; ir++) {
    for (ic = ir + noffdiag + 1; ic < w.upbound; ic++) {
      b[ic + shift][ir + shift] = 0.;
      b[ir + shift][ic + shift] = 0.;
    }
  }
}

/*
void umolProb::prepABmat() {                          // --- prepABmat ---
  int ir, jc, irStart, irEnd;
  for (jc = 0; jc < psiz; jc++) {
    irStart = jc - noffdiag; irEnd = jc;
    if (irStart < 0) { irStart = 0; }
    for (ir = irStart; ir <= irEnd; ir++) {
      ab[jc][ir - jc + noffdiag] = b[jc][ir];
    }
  }
}
*/

void umolProb::findLargeNegEigen() {          // --- findLargeNegEigen ---
  int i;
  posLNE = esiz - 1;
  for (i = esiz - 1; i >= 0; i--) {
    if (v[i] < 0.) { posLNE = i; break; }
  }
}

                                                        // --- normG1W ---
double umolProb::normG1W(well &w, int pos0, double *g) {
  int i, imat;
  double sumg = 0.;
  for (i = w.lowest; i < w.upbound; i++) {
    imat = pos0 + i - w.lowest;
    sumg += g[imat] * s[imat];
  }
  return sumg;
}

double umolProb::normGMW(listWells &lw, double *g) {    // --- normGMW ---
  int posw, pos0;
  double totsumg = 0.;
  for (posw = 0; posw < lw.size(); posw++) {
    pos0 = lw[posw].posBmat;
    totsumg += (lw[posw].current.fpop = normG1W(lw[posw], pos0, g));
  }
  for (posw = 0; posw < lw.size(); posw++)
    { lw[posw].current.fpop /= totsumg; }
  return totsumg;
}

                                                     // --- integkEg1W ---
void umolProb::integkEg1W(well &w, int pos0, double *g, double gnorm) {
  int i, ch, shift = pos0 - w.lowest, irat, imat;
  double sumdg;
  vector<double> sumkg;

  if (probtype == Eigen) { w.current.keigen = -v[posLNE]; }
  sumdg = 0.; sumkg.clear();
  for (ch = 0; ch < w.nchan; ch++) { sumkg.push_back(0.); }
  for (i = w.lowest; i < w.upbound; i++) {
    imat = i + shift; irat = i - w.nnonrg;
    if (irat >= 0) {
      for (ch = 0; ch < w.nchan; ch++)
        { sumkg[ch] += g[imat] * s[imat] * w.rate[ch][irat]; }
    }
    sumdg += g[imat] * s[imat] * d[imat];
  }
  w.current.ktot = 0.;
  w.current.k.clear();
  for (ch = 0; ch < w.nchan; ch++) {
    w.current.k.push_back(sumkg[ch] / gnorm);
    w.current.ktot += sumkg[ch] / gnorm;
  }
  w.current.kstab = sumdg / gnorm;
  w.current.ktot += sumdg / gnorm;
  w.current.f.clear();
  for (ch = 0; ch < w.nchan; ch++) {
    w.current.f.push_back(w.current.k[ch] / w.current.ktot);
  }
  w.current.fstab = w.current.kstab / w.current.ktot;
}

                                                     // --- integkEgMW ---
void umolProb::integkEgMW(listWells &lw, double *g, double tgnorm) {
  int posw, pos0, ch;
  if (probtype == Eigen) { lw.keigen = -v[posLNE]; }
  lw.koutTot = 0.;
  for (posw = 0; posw < lw.size(); posw++) {
    pos0 = lw[posw].posBmat;
    integkEg1W(lw[posw], pos0, g, tgnorm);
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1) { lw.koutTot += lw[posw].current.k[ch]; }
    }
    if (probtype == LinEq) { lw.koutTot += lw[posw].current.kstab; }
  }
}

double umolProb::calcDGdt() {                          // --- calcDGdt ---
  // z[0] = b[][] * r[0][]
  int i, j;
  double s, normabs = 0.;
  for (i = 0; i < psiz; i++) {
    s = 0.;
    for (j = 0; j < psiz; j++) { s += b[j][i] * r[0][j]; }
    z[0][i] = s;
    normabs += fabs(s);
  }
  return normabs;
}

void umolProb::addDGstep(double nabs, double stp) {   // --- addDGstep ---
  double norm = 0.;
  int i;
  for (i = 0; i < psiz; i++) {
    r[0][i] += z[0][i] / nabs * stp;
    if (r[0][i] < 0.) { r[0][i] = 0.; }
    norm += r[0][i] * s[i];
  }
  for (i = 0; i < psiz; i++) { r[0][i] /= norm; }
}

