/*========================================================================
     "ssulibb" - basic library functions
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2009 by A. Miyoshi, Univ. Tokyo
                                  created: Sept. 22, 2009 (from gpoplibf)
                              last edited: Dec.   8, 2009
========================================================================*/

#include "ssulibb.h"

/*========================================================================
     General Purpose Funtions
========================================================================*/

/*..... show revision number ...........................................*/

void showRevision() { cout << REVSTR << endl; }

/*..... get base part from a filename ..................................*/

string getBasePart(string flname) {
  string basname;
  string::size_type p;
  if ((p = flname.rfind('.')) == string::npos) {
    basname = flname;
  } else {
    basname = flname.substr(0, p);
  }
  return basname;
}

/*..... get extension part from a filename .............................*/

string getExtPart(string flname) {
  string extname;
  string::size_type p;
  if ((p = flname.rfind('.')) == string::npos) {
    extname.erase();
  } else {
    extname = flname.substr(p + 1);
  }
  return extname;
}

/*..... get a line from stream with removal of comment part ............*/

int getLineRmCmm(istream &inStrm, string &sLine, const string &cmmSyms) {
  int ics;
  char cs;
  string::size_type p;
  do {
    if (getLine(inStrm, sLine) == ERRE) { return ERRE; }
    for (ics = 0; ics < cmmSyms.length(); ics++) {
      cs = cmmSyms[ics];
      if ((p = sLine.find(cs)) != string::npos)
        { sLine = sLine.substr(0, p); }
    }
    sLine = trim(sLine);
  } while (sLine == "");
  return NORM;
}

/*..... get a line from stream .........................................*/

int getLine(istream &inStrm, string &sLine) {
  static char inBuf[MAXLN1L];
  char c;
  int p = 0;
  sLine = "";
  if (inStrm.get(c).eof()) { return ERRE; }
  do {
    if (c == '\n') { break; }
    if (c != '\r') {
      inBuf[p] = c; p++;
      if (p == (MAXLN1L - 1)) { break; }
    }
  } while (!inStrm.get(c).eof());
  inBuf[p] = '\0';
  sLine = inBuf;
  return NORM;
}

/*..... get a line from stream and trim spaces .........................*/

int getLineTrim(istream &inStrm, string &sLine) {
  if (getLine(inStrm, sLine) == ERRE) { return ERRE; }
  sLine = trim(sLine);
  return NORM;
}

/*..... convert string to upper case ...................................*/

void strAllToUpper(string &s) {
  long ls = s.length(), p;
  for (p = 0; p < ls; p++) { s[p] = toupper(s[p]); }
}

/*..... convert string to lower case ...................................*/

void strAllToLower(string &s) {
  long ls = s.length(), p;
  for (p = 0; p < ls; p++) { s[p] = tolower(s[p]); }
}

/*..... trim preceding & succeeding spaces in the string ...............*/

string ltrim(string s) {
  long p, first = s.length();
  for (p = 0; p < s.length(); p++) {
    if ((s[p] != ' ') && (s[p] != '\t')) { first = p; break; }
  }
  return s.substr(first);
}

string rtrim(string s) {
  long p, last = -1;
  for (p = s.length() - 1; p >= 0; p--) {
    if ((s[p] != ' ') && (s[p] != '\t')) { last = p; break; }
  }
  return s.substr(0, last + 1);
}

string trim(string s) {
  string rs = s;
  rs = ltrim(rs); rs = rtrim(rs);
  return rs;
}

/*..... split string to tokens .........................................*/

vector<string> splitTokens(string &s, const string &spr) {
  long pos, lastpos = -1, slen = s.length();
  string tok;
  vector<string> rs;
  rs.clear();
  for (pos = 0; pos < slen; pos++) {
    if (spr.find(s[pos]) != string::npos) {
      if (lastpos + 1 < pos) {
        tok = s.substr(lastpos + 1, pos - lastpos - 1);
        rs.push_back(tok);
      }
      lastpos = pos;
    }
  }
  if (lastpos < slen - 1) {
    tok = s.substr(lastpos + 1);
    rs.push_back(tok);
  }
  return rs;
}

/*..... string formatting ..............................................*/

string format(int v, int len, char filc) {
  ostringstream os;
  os << setw(len) << setfill(filc) << v;
  return os.str();
}

/*..... prepare a string of list of indices ............................*/

string strListIdx(vector<int> &lst) {
  int i, n;
  ostringstream oss;
  n = lst.size();
  for (i = 0; i < n; i++) {
    if (i > 0) { oss << '-'; }
    oss << (lst[i] + 1);
  }
  return oss.str();
}

/*..... prepare a string of a boolean value ............................*/

string strBoolVal(int &vBln) {
  string rets;
  if (vBln) { rets = "true"; } else { rets = "false"; }
  return rets;
}

/*..... get a boolean value from a string ..............................*/

int valBoolStr(string s) {
  int retv;
  strAllToUpper(s);
  if (s == "TRUE") { retv = true; } else { retv = false; }
  return retv;
}

/*========================================================================
     Common Library Functions
========================================================================*/

// --- Parameter File Input Functions ------------------------------------

/*----- get a line from stream removing comment part (#!) --------------*/

int getLinePar(istream &inStrm, string &sLine) {
  return getLineRmCmm(inStrm, sLine, "#!");
}

/*----- get a line and check for matching ------------------------------*/

int getlParMatch(istream &inStrm, const string &ref) {
  string sLine;
  if (getLinePar(inStrm, sLine) == ERRE) { return ERRE; }
  if (sLine != ref) { return ERRE; }
  return NORM;
}

/*----- get a line and token-splitted strings --------------------------*/
int getlParToks(istream &inStrm, vector<string> &toks, int &ntoks,
 string &sLine) {
  if (getLinePar(inStrm, sLine) == ERRE) { return ERRE; }
  toks = splitTokens(sLine, " ,\t"); ntoks = toks.size();
  if (ntoks <= 0) { return ERRE; }
  return NORM;
}

/*----- get a line, token-splitted strings, and key --------------------*/

int getlParToksKey(istream &inStrm, string &key, vector<string> &toks,
 int &ntoks, string &sLine) {
  if (getlParToks(inStrm, toks, ntoks, sLine) == ERRE) { return ERRE; }
  key = toks[0];
  return NORM;
}

