/*========================================================================
     "numlibChb" - regression to Chebyshev polynomials
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2009 by A. Miyoshi, Univ. Tokyo
                                  created: Dec.   8, 2009
                              last edited: Dec.  11, 2009
========================================================================*/

#include "numlibChb.h"

/*========================================================================
     Rate coefficients regression to Chebyshev polynomials
========================================================================*/

/*..... Chebyshev polynomial of degree n ...............................*/

double cheb(int n, double x) { return cos(n * acos(x)); }

/*===== Chebyshev polynomial fitting of TP data ========================*/

chebyTPfit::chebyTPfit() { clear(); }              // --- initializers ---

chebyTPfit::chebyTPfit(int npT, int npP, double Ts, double Te,
 double Ps, double Pe)
  { init(npT, npP, Ts, Te, Ps, Pe); }

void chebyTPfit::clear() {
  basicLinLsSq::clear();
  nParT = 0; nParP = 0;
  tkdat.clear();
}

int chebyTPfit::init(int npT, int npP, double Ts, double Te,
 double Ps, double Pe) {
  clear();
  if (basicLinLsSq::init(npT * npP) == ERRE) { return ERRE; }
  if ((npT < 1) || (npP < 1))
    { cout << "number of parameters must be > 0.\n"; return ERRE; }
  if ((Ts <= 0.) || (Ts >= Te))
    { cout << "invalid T-range.\n"; return ERRE; }
  if ((Ps <= 0.) || (Ps >= Pe))
    { cout << "invalid P-range.\n"; return ERRE; }
  nParT = npT; nParP = npP;
  Tmin = Ts; Tmax = Te; Pmin = Ps; Pmax = Pe;
  invTmin = 1. / Ts; invTmax = 1. / Te;
  logPmin = log(Ps); logPmax = log(Pe);
  return NORM;
}

                                           // --- add one k(T,P) datum ---
int chebyTPfit::addRateTP(double k, double T, double P) {
  int it, ip;
  double logk, Ttil, Ptil;
  vector<double> x;
  transDatum tkd;
  if (k <= 0.) { cout << "invalid k.\n"; return ERRE; }
  if ((T < Tmin) || (T > Tmax)) { cout << "invalid T.\n"; return ERRE; }
  if ((P < Pmin) || (P > Pmax)) { cout << "invalid P.\n"; return ERRE; }
  logk = log10(k); Ttil = Ttilde(T); Ptil = Ptilde(P);
  x.clear();
  for (it = 0; it < nParT; it++) {
    for (ip = 0; ip < nParP; ip++)
      { x.push_back(cheb(it, Ttil) * cheb(ip, Ptil)); }
  }
  if (basicLinLsSq::addDatum(logk, x) == ERRE) { return ERRE; }
  tkd.logk = logk; tkd.Ttil = Ttil; tkd.Ptil = Ptil;
  tkdat.push_back(tkd);
  return NORM;
}

                               // --- add one k(T,P) datum by kDatumTP ---
int chebyTPfit::addRateTP(kDatumTP &kd) {
  return addRateTP(kd.k, kd.T, kd.P);
}

                                         // --- add k(T,P)'s in arrays ---
int chebyTPfit::addRateTParray(vector<double> &k, vector<double> &T,
 vector<double> &P) {
  int jd, nd = k.size();
  if (T.size() < nd) { nd = T.size(); }
  if (P.size() < nd) { nd = P.size(); }
  for (jd = 0; jd < nd; jd++) {
    if (addRateTP(k[jd], T[jd], P[jd]) == ERRE) { return ERRE; }
  }
  return NORM;
}

                               // --- add k(T,P)'s by vector<kDatumTP> ---
int chebyTPfit::addRateTParray(vector<kDatumTP> &vkd) {
  int jd, nd = vkd.size();
  for (jd = 0; jd < nd; jd++) {
    if (addRateTP(vkd[jd]) == ERRE) { return ERRE; }
  }
  return NORM;
}

                                              // --- add k(T,P) matrix ---
int chebyTPfit::addRateTPmatrix(vector<vector<double> > &k,
 vector<double> &T, vector<double> &P) {
  int jt, jp, nt = T.size(), np = P.size();
  if (k.size() < nt)
    { cout << "incosistent sizes for k and T.\n"; return ERRE; }
  for (jt = 0; jt < nt; jt++) {
    if (k[jt].size() < np)
      { cout << "incosistent sizes for k and P.\n"; return ERRE; }
    for (jp = 0; jp < np; jp++) {
      if (addRateTP(k[jt][jp], T[jt], P[jp]) == ERRE) { return ERRE; }
    }
  }
  return NORM;
}

int chebyTPfit::solve() {                 // --- solve linear equation ---
  int id, nd = tkdat.size();
  double ad;
  if (basicLinLsSq::solve_blls(parA, stdA, corcoefA, sdk) == ERRE)
    { return ERRE; }
  madk = 0.; maxadk = 0.;
  for (id = 0; id < nd; id++) {
    ad = fabs(calfit(tkdat[id].Ttil, tkdat[id].Ptil) - tkdat[id].logk);
    madk += ad;
    if (ad > maxadk) { maxadk = ad; }
  }
  madk /= nd;
  return NORM;
}

/*----- private functions -----------------------------------------*/

                                          // --- calculate fitted logk ---
double chebyTPfit::calfit(double Tt, double Pt) {
  int it, ip, ia = 0;
  double logk = 0.;
  for (it = 0; it < nParT; it++) {
    for (ip = 0; ip < nParP; ip++) {
      logk += parA[ia] * cheb(it, Tt) * cheb(ip, Pt);
      ia++;
    }
  }
  return logk;
}

double chebyTPfit::Ttilde(double T)        // --- convert T to T-tilde ---
  { return (2. / T - invTmax - invTmin) / (invTmax - invTmin); }

double chebyTPfit::Ptilde(double P)        // --- convert P to P-tilde ---
  { return (2. * log(P) - logPmax - logPmin) / (logPmax - logPmin); }

