/*========================================================================
     "fitcheb" - fits RRKM results to Chebyshev polynomials
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2009 by A. Miyoshi, Univ. Tokyo
                                  created: Dec.   8, 2009
                              last edited: Dec.  14, 2009
--------------------------------------------------------------------------
 usage: fitcheb basename(.ext)
--------------------------------------------------------------------------
 - If no extension is given, default extension '.inp' is assumed
 - output is written in [basename]_fitcheb_out.csv
========================================================================*/

#include "ssulibb.h"
#include "numlibChb.h"

int calcOnePar(chebyTPfit *chbf, vector<kDatumTP> &vkd, double Tmin,
 double Tmax, double Pmin, double Pmax, int nParT, int nParP);
int outputCSV(string csvfln, chebyTPfit *chbf);
int scanPars(string diafln, chebyTPfit *chbf, vector<kDatumTP> &vkd,
 double Tmin, double Tmax, double Pmin, double Pmax);

/*===== main function ==================================================*/

int main(int argc, char *argv[]) {
  int ntkns, nParT = -1, nParP = -1, endTable, scan = false;
  double Tmin = -1., Tmax, Pmin = -1., Pmax;
  string inpfln, basfln, extfln, csvfln, diafln;
  string key, slin;
  vector<string> tkns;
  ifstream inf;
  kDatumTP kTP;
  vector<kDatumTP> vkd;
  chebyTPfit *chbf = new chebyTPfit;

  cout << "fitcheb - "; showRevision();
  if (argc != 2) { cout << "usage: fitcheb basename(.ext)\n"; return 1; }
  inpfln = argv[1];
  basfln = getBasePart(inpfln); extfln = getExtPart(inpfln);
  if (extfln == "") { inpfln = basfln + ".inp"; }
  csvfln = basfln + "_fitcheb_out.csv";
  diafln = basfln + "_fitcheb_diag.csv";

  // --- read in input
  inf.open(inpfln.c_str());
  if (!inf) { cout << "failed to open [" << inpfln << "].\n"; return 1; }
  vkd.clear();
  while (getlParToksKey(inf, key, tkns, ntkns, slin) == NORM) {
    if (key == "numPars") {
      if (ntkns < 3) {
        cout << "Insufficient number of values for key [numPars].\n";
        return 1;
      }
      nParT = atoi(tkns[1].c_str()); nParP = atoi(tkns[2].c_str());
    } else if (key == "tempCheb") {
      if (ntkns < 3) {
        cout << "Insufficient number of values for key [tempCheb].\n";
        return 1;
      }
      Tmin = atof(tkns[1].c_str()); Tmax = atof(tkns[2].c_str());
      if ((Tmin >= Tmax) || (Tmin <= 0.))
        { cout << "Invalid values for key [tempCheb].\n"; return 1; }
    } else if (key == "pressCheb") {
      if (ntkns < 3) {
        cout << "Insufficient number of values for key [pressCheb].\n";
        return 1;
      }
      Pmin = atof(tkns[1].c_str()); Pmax = atof(tkns[2].c_str());
      if ((Pmin >= Pmax) || (Pmin <= 0.))
        { cout << "Invalid values for key [pressCheb].\n"; return 1; }
    } else if (key == "TPk_Table{") {
      endTable = false;
      while (getlParToks(inf, tkns, ntkns, slin) == NORM) {
        if (tkns[0] == "}") { endTable = true; break; }
        if (ntkns < 3) {
          cout << "Insufficient number of values for k-table.\n";
          return 1;
        }
        kTP.T = atof(tkns[0].c_str()); kTP.P = atof(tkns[1].c_str());
        kTP.k = atof(tkns[2].c_str());
        vkd.push_back(kTP);
      }
      if (!endTable)
        { cout << "Unexpected end of table data.\n"; return 1; }
    } else { cout << "Warning: Invalid key [" << key << "].\n"; }
  }
  inf.close();

  // --- check for input
  if (Tmin < 0.) { cout << "No [tempCheb] input found.\n"; return 1; }
  if (Pmin < 0.) { cout << "No [pressCheb] input found.\n"; return 1; }
  if ((nParT <= 0) || (nParP <= 0)) { scan = true; }

  // --- do calculation
  if (!scan) {
    if (calcOnePar(chbf, vkd, Tmin, Tmax, Pmin, Pmax, nParT, nParP) == ERRE)
      { return 1; }
    if (outputCSV(csvfln, chbf) == ERRE) { return 1; }
  } else {
    if (scanPars(diafln, chbf, vkd, Tmin, Tmax, Pmin, Pmax) == ERRE)
      { return 1; }
  }
  return 0;
}

/*----- calcaulate for a set of (nParT, nParP) -------------------------*/

int calcOnePar(chebyTPfit *chbf, vector<kDatumTP> &vkd, double Tmin,
 double Tmax, double Pmin, double Pmax, int nParT, int nParP) {
  if (chbf->init(nParT, nParP, Tmin, Tmax, Pmin, Pmax) == ERRE)
    { return ERRE; }
  if (chbf->addRateTParray(vkd) == ERRE) { return ERRE; }
  if (chbf->solve() == ERRE) { return ERRE; }
  return NORM;
}

/*----- output results to a csv file -----------------------------------*/

int outputCSV(string csvfln, chebyTPfit *chbf) {
  int it, ip, ia, nt = chbf->nParT, np = chbf->nParP;
  ofstream cof;
  cof.open(csvfln.c_str());
  if (!cof)
    { cout << "failed to open [" << csvfln << "].\n"; return ERRE; }
  cof << "nParT," << chbf->nParT << endl;
  cof << "nParP," << chbf->nParP << endl;
  cof << "STD(log10k)," << chbf->sdk << endl;
  cof << "MAD(log10k)," << chbf->madk << endl;
  cof << "MAXAD(log10k)," << chbf->maxadk << endl;
  cof << "a(iT_iP),,iP\n,";
  for (ip = 0; ip < np; ip++) { cof << "," << (ip + 1); }
  cof << endl;
  ia = 0;
  for (it = 0; it < nt; it++) {
    if (it == 0) { cof << "iT"; }
    cof << "," << (it + 1);
    for (ip = 0; ip < np; ip++) { cof << "," << chbf->parA[ia]; ia++; }
    cof << endl;
  }
  cof.close();
  return NORM;
}

/*----- scan nParT and nParP -------------------------------------------*/

int scanPars(string diafln, chebyTPfit *chbf, vector<kDatumTP> &vkd,
 double Tmin, double Tmax, double Pmin, double Pmax) {
  int nt, np, maxntnp = basicLinLsSq::MXNPAR, maxn;
  vector<double> vsd, vmad, vmaxad;
  vector<vector<double> > vvsd, vvmad, vvmaxad;
  ofstream dof;
  if (maxntnp > vkd.size()) { maxntnp = vkd.size(); }
  maxn = sqrt((double)maxntnp);
  for (nt = 2; nt <= maxn; nt++) {
    vsd.clear(); vmad.clear(); vmaxad.clear();
    for (np = 2; np <= maxn; np++) {
      if (calcOnePar(chbf, vkd, Tmin, Tmax, Pmin, Pmax, nt, np) == ERRE) {
        vsd.push_back(-1.); vmad.push_back(-1.); vmaxad.push_back(-1.);
      } else {
        vsd.push_back(chbf->sdk); vmad.push_back(chbf->madk);
        vmaxad.push_back(chbf->maxadk);
      }
    }
    vvsd.push_back(vsd); vvmad.push_back(vmad); vvmaxad.push_back(vmaxad);
  }
  dof.open(diafln.c_str());
  if (!dof)
    { cout << "failed to open [" << diafln << "].\n"; return ERRE; }
  dof << "STD(log10k),,nParP\n,";
  for (np = 2; np <= maxn; np++) { dof << "," << np; }
  dof << endl;
  for (nt = 2; nt <= maxn; nt++) {
    if (nt == 2) { dof << "nParT"; }
    dof << "," << nt;
    for (np = 2; np <= maxn; np++) {
      dof << ",";
      if (vvsd[nt - 2][np - 2] >= 0.) { dof << vvsd[nt - 2][np - 2]; }
    }
    dof << endl;
  }
  dof << "\nMAD(log10k),,nParP\n,";
  for (np = 2; np <= maxn; np++) { dof << "," << np; }
  dof << endl;
  for (nt = 2; nt <= maxn; nt++) {
    if (nt == 2) { dof << "nParT"; }
    dof << "," << nt;
    for (np = 2; np <= maxn; np++) {
      dof << ",";
      if (vvmad[nt - 2][np - 2] >= 0.) { dof << vvmad[nt - 2][np - 2]; }
    }
    dof << endl;
  }
  dof << "\nMAXAD(log10k),,nParP\n,";
  for (np = 2; np <= maxn; np++) { dof << "," << np; }
  dof << endl;
  for (nt = 2; nt <= maxn; nt++) {
    if (nt == 2) { dof << "nParT"; }
    dof << "," << nt;
    for (np = 2; np <= maxn; np++) {
      dof << ",";
      if (vvmaxad[nt - 2][np - 2] >= 0.) { dof << vvmaxad[nt - 2][np - 2]; }
    }
    dof << endl;
  }
  dof.close();
  return NORM;
}

