/*========================================================================
     "diseig" - steady-state dissociation eigen problem solver
                 for multiple-well system
      This is a part of "ssumes" -
        Steady-State Unimoluclar Master Equation Solver
             Copyright (c) 2002-2009 by A. Miyoshi, Univ. Tokyo
                                  created: Sept. 29, 2009
                              last edited: Nov.  14, 2009
--------------------------------------------------------------------------
 usage: diseig basename(.ext) (-e)
--------------------------------------------------------------------------
 - If no extension is given, default extension '.inp' is assumed
 - output is written in [basename]_diseig_out.csv
 - HPL rate is written in [basename]_diseig_hpl.csv
 - when -e option is set, eigenvalues are written
    in [basename]_diseig_eig.csv
 - when 'output vector' is set, steady-state vectors are written
    in [basename]_diseig_vec.csv
========================================================================*/

#include "ssulibc.h"

void outputHeader(ostream &os, listWells &lw);
void outputResults(ostream &os, listWells &lw);

/*===== main function ==================================================*/

int main(int argc, char *argv[]) {
  int prtEigen = false, inWell, ntkns, idxT, idxP, posw, outVec = false;
  int nReactWell = 0, err3set = false, outHPL = false, HPLonly = false;
  int itkn;
  double err3inp;
  string ctrlfln, basfln, extfln, csvfln, hplfln, eigfln, vecfln;
  string opt, slin, key;
  ifstream inf;
  ofstream cof, hof, ecf, vcf;
  vector<string> tkns;
  tempList tlst;
  pressList plst;
  connex cnx;
  well molecw;
  listWells lstw;
  umolProb *ump = new umolProb;   // large object cannot be an autovariable

  cout << "diseig - "; showRevision();
  if ((argc < 2) || (argc > 3))
    { cout << "usage: diseig basename(.ext) (-e)\n"; return 1; }
  if (argc > 2) {
    opt = argv[2];
    if (opt == "-e") { prtEigen = true; }
    else { cout << "ignored invalid option [" << opt << "]\n"; }
  }
  ctrlfln = argv[1];
  basfln = getBasePart(ctrlfln); extfln = getExtPart(ctrlfln);
  if (extfln == "") { ctrlfln = basfln + ".inp"; }
  csvfln = basfln + "_diseig_out.csv";
  hplfln = basfln + "_diseig_hpl.csv";
  eigfln = basfln + "_diseig_eig.csv";
  vecfln = basfln + "_diseig_vec.csv";
  ump->probtype = umolProb::Eigen;

  // --- read in control input
  inf.open(ctrlfln.c_str());
  if (!inf) { cout << "failed to open [" << ctrlfln << "].\n"; return 1; }
  inWell = false;
  while (getlParToksKey(inf, key, tkns, ntkns, slin) == NORM) {
    if (inWell) {
      if (key == "}") {
        if (lstw.add(molecw) == ERRE) {
          cout << "Duplicated well-index [" << molecw.index << "].\n";
          return 1;
        }
        inWell = false;
      } else if (key == "index") {
        if (ntkns < 2) { cout << "No value for key [index].\n"; return 1; }
        molecw.index = atoi(tkns[1].c_str());
      } else if (key == "filename") {
        if (ntkns < 2) { cout << "No value for key [filename].\n"; return 1; }
        molecw.fname = tkns[1];
      } else if (key == "offset") {
        if (ntkns < 2) { cout << "No value for key [offset].\n"; return 1; }
        molecw.offset = atoi(tkns[1].c_str());
      } else if (key == "connect") {
        if (ntkns < 3)
          { cout << "Two values needed for key [connect].\n"; return 1; }
        cnx.toWellIdx = atoi(tkns[1].c_str());
        cnx.chan = atoi(tkns[2].c_str()) - 1;
        molecw.connect.push_back(cnx);

/*     --- disabled 12-Nov-2009

      } else if (key == "thRhoGrad") {
        if (ntkns < 2) { cout << "No value for key [thRhoGrad].\n"; return 1; }
        molecw.thRhoGrad = atof(tkns[1].c_str());
      } else if (key == "redExpon") {
        if (ntkns < 2) { cout << "No value for key [redExpon].\n"; return 1; }
        molecw.redExpFix = atof(tkns[1].c_str());
      } else if (key == "reactant") {
        molecw.isReactant = true;
*/
      } else { cout << "Warning: Invalid key [" << key << "].\n"; }
    } else {
      if (key == "well{") {
        molecw.clear();
        molecw.index = lstw.autoIndex();
        inWell = true;
      } else if (key.substr(0, 4) == "temp") {      // temperature list input
        if (tlst.addFrom(slin) == ERRE) { return 1; }
      } else if (key.substr(0, 5) == "press") {     // pressure list input
        if (plst.addFrom(slin) == ERRE) { return 1; }
      } else if (key == "topCut") {
        if (ntkns < 2) { cout << "No value for key [topCut].\n"; return 1; }
        lstw.topCut = atoi(tkns[1].c_str());
      } else if (key == "err3") {
        if (ntkns < 2) { cout << "No value for key [err3].\n"; return 1; }
        err3inp = atof(tkns[1].c_str()); err3set = true;
      } else if (key == "verbose") {
        if (ntkns < 2) { cout << "No value for key [verbose].\n"; return 1; }
        ump->verbose = atoi(tkns[1].c_str());
      } else if (key == "output") {
        if (ntkns < 2) { cout << "Warning: No value for key [output].\n"; }
        else {
          for (itkn = 1; itkn < ntkns; itkn++) {
            if (tkns[itkn] == "vector") {
              outVec = true;
            } else if (tkns[itkn] == "HPL") {
              outHPL = true;
            } else if (tkns[itkn] == "HPLonly") {
              outHPL = true; HPLonly = true;
            } else {
              cout << "Warning: Invalid value [" << tkns[1]
                   << "] for key [output].\n";
            }
          }
        }
      } else { cout << "Warning: Invalid key [" << key << "].\n"; }
    }
  }
  inf.close();

  // --- check for reactant well
  for (posw = 0; posw < lstw.size(); posw++) {
    if (lstw[posw].isReactant) { nReactWell++; lstw.posReactWell = posw; }
  }
  if (nReactWell > 1) { cout << "Too may reactant-wells.\n"; return 1; }

  // --- read in Unimol RRKMTH outputs
  if (lstw.readUnimol() == ERRE) { return 1; }
  if (err3set) { lstw.setErr3(err3inp); }
  if (tlst.empty()) { lstw.mergedTlist(tlst); }
  reverse(tlst.begin(), tlst.end());
  if (plst.empty()) { lstw.mergedPlist(plst); }
  reverse(plst.begin(), plst.end());
  lstw.printParETP(cout, ump->verbose);

  // --- do calculations
  if (ump->initMW(lstw) == ERRE) { return 1; }
   // --- HPL rate
  if (outHPL) {
    hof.open(hplfln.c_str());
    if (!hof) { cout << "failed to open [" << hplfln << "].\n"; return 1; }
    outputHeader(hof, lstw);
    for (idxT = 0; idxT < tlst.size(); idxT++) {
      ump->calcHPLMW(lstw, tlst[idxT]);
      hof << ump->T << ",inf";
      outputResults(hof, lstw);
    }
    hof.close();
  }
  if (HPLonly) { return 0; }
  cof.open(csvfln.c_str());
  if (!cof) { cout << "failed to open [" << csvfln << "].\n"; return 1; }
  if (prtEigen) {
    ecf.open(eigfln.c_str());
    if (!ecf) { cout << "failed to open [" << eigfln << "].\n"; return 1; }
    ecf << "T[K],P[Torr],eigenvalues\n";
  }
  if (outVec) {
    vcf.open(vecfln.c_str());
    if (!vcf) { cout << "failed to open [" << vecfln << "].\n"; return 1; }
  }
   // --- main loop
  outputHeader(cof, lstw);
  for (idxT = 0; idxT < tlst.size(); idxT++) {
    for (idxP = 0; idxP < plst.size(); idxP++) {
      cout << "T = " << tlst[idxT] << " [K], p = " << plst[idxP] << " [Torr]:\n";
      if (ump->setMatMW(lstw, tlst[idxT], plst[idxP]) == ERRE) { return 1; }
//    if (ump->solveEigenMW(lstw) == ERRE) { return 1; }
      if (ump->solveEigenMWdsyevr(lstw) == ERRE) { return 1; }
      cof << ump->T << ',' << ump->p;
      outputResults(cof, lstw);
      if (prtEigen) {
        ecf << ump->T << ',' << ump->p << ',';
        ump->outEigVal(ecf, 10);
      }
      if (outVec) { ump->outputVectorMW(vcf, lstw); }
    }
  }
  if (outVec) { vcf.close(); }
  if (prtEigen) { ecf.close(); }
  cof.close();
  return 0;
}

/*----- output header of a csv file ------------------------------------*/

void outputHeader(ostream &os, listWells &lw) {
  int ch, posw, toposw;
   // - 1st line
  os << ",,out-going-rates";
  for (ch = 0; ch < lw.nTotOCh; ch++) { os << ','; }
  if (lw.nTotOCh > 0) { os << ",out-going-fractions"; }
  for (ch = 0; ch < lw.nTotOCh - 1; ch++) { os << ','; }
  if (lw.nTotCh - lw.nTotOCh > 0) { os << ",internal-rates"; }
  for (ch = 0; ch < lw.nTotCh - lw.nTotOCh - 1; ch++) { os << ','; }
  os << ",internal-populations";
  for (posw = 1; posw < lw.size(); posw++) { os << ','; }
  os << endl;
   // - 2nd line
  os << ',';
  for (posw = 0; posw < lw.size(); posw++) {
    if (lw[posw].nOutCh > 0) {
      os << ",well" << lw[posw].index;
      if (lw[posw].isReactant) { os << "(reactant)"; }
    }
    for (ch = 0; ch < lw[posw].nOutCh - 1; ch++) { os << ','; }
  }
  os << ",ktot";
  for (posw = 0; posw < lw.size(); posw++) {
    if (lw[posw].nOutCh > 0) {
      os << ",well" << lw[posw].index;
      if (lw[posw].isReactant) { os << "(reactant)"; }
    }
    for (ch = 0; ch < lw[posw].nOutCh - 1; ch++) { os << ','; }
  }
  for (posw = 0; posw < lw.size(); posw++) {
    if (lw[posw].nchan - lw[posw].nOutCh > 0) {
      os << ",well" << lw[posw].index;
      if (lw[posw].isReactant) { os << "(reactant)"; }
    }
    for (ch = 0; ch < lw[posw].nchan - lw[posw].nOutCh - 1; ch++)
      { os << ','; }
  }
  for (posw = 0; posw < lw.size(); posw++) {
    os << ",well" << lw[posw].index;
    if (lw[posw].isReactant) { os << "(reactant)"; }
  }
  os << endl;
   // - 3rd line
  os << "T[K],P[Torr]";
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1) { os << ",k" << (ch + 1); }
    }
  }
  os << ',';
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1) { os << ",f" << (ch + 1); }
    }
  }
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if ((toposw = lw[posw].chInf[ch]) >= 0)
        { os << ",k" << (ch + 1) << "(to" << lw[toposw].index << ')'; }
    }
  }
  for (posw = 0; posw < lw.size(); posw++) { os << ','; }
  os << endl;
}

/*----- output results to a csv file -----------------------------------*/

void outputResults(ostream &os, listWells &lw) {
  int posw, ch;
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1)
        { os << ',' << lw[posw].current.k[ch]; }
    }
  }
  os << ',' << lw.koutTot;
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] == -1)
        { os << ',' << lw[posw].current.k[ch] / lw.koutTot; }
    }
  }
  for (posw = 0; posw < lw.size(); posw++) {
    for (ch = 0; ch < lw[posw].nchan; ch++) {
      if (lw[posw].chInf[ch] >= 0)
        { os << ',' << lw[posw].current.k[ch]; }
    }
  }
  for (posw = 0; posw < lw.size(); posw++)
    { os << ',' << lw[posw].current.fpop; }
  os << endl;
}

